﻿using System;
using System.Collections.Generic;
using System.Windows.Input;

namespace calboxgui
{
    /// <summary>
    /// This class allows delegating the commanding logic to methods passed as parameters,
    /// and enables a View to bind commands to objects that are not part of the element tree.
    /// </summary>
    public class DelegateCommand : ICommand
    {
        #region Constructors

        /// <summary>
        /// Initializes a new instance of the <see cref="DelegateCommand"/> class.
        /// </summary>
        /// <param name="executeMethod">The execute method.</param>
        public DelegateCommand( Action executeMethod )
            : this( executeMethod, null, false ) { }

        /// <summary>
        /// Initializes a new instance of the <see cref="DelegateCommand"/> class.
        /// </summary>
        /// <param name="executeMethod">The execute method.</param>
        /// <param name="canExecuteMethod">The can execute method.</param>
        public DelegateCommand( Action executeMethod, Func<bool> canExecuteMethod )
            : this( executeMethod, canExecuteMethod, false ) { }


        /// <summary>
        /// Initializes a new instance of the <see cref="DelegateCommand"/> class.
        /// </summary>
        /// <param name="executeMethod">The execute method.</param>
        /// <param name="canExecuteMethod">The can execute method.</param>
        /// <param name="isAutomaticRequeryDisabled">if set to <c>true</c> [is automatic requery disabled].</param>
        /// <exception cref="ArgumentNullException"><c>executeMethod</c> is null.</exception>
        public DelegateCommand( Action executeMethod, Func<bool> canExecuteMethod, bool isAutomaticRequeryDisabled )
        {
            if( executeMethod == null )
            {
                throw new ArgumentNullException( "executeMethod" );
            }

            this._executeMethod = executeMethod;
            this._canExecuteMethod = canExecuteMethod;
            this._isAutomaticRequeryDisabled = isAutomaticRequeryDisabled;
        }

        #endregion

        #region Public Methods

        /// <summary>
        /// Property to enable or disable CommandManager's automatic requery on this command
        /// </summary>
        public bool IsAutomaticRequeryDisabled
        {
            get { return this._isAutomaticRequeryDisabled; }
            set
            {
                if( this._isAutomaticRequeryDisabled != value )
                {
                    if( value )
                    {
                        CommandManagerHelper.RemoveHandlersFromRequerySuggested( this._canExecuteChangedHandlers );
                    }
                    else
                    {
                        CommandManagerHelper.AddHandlersToRequerySuggested( this._canExecuteChangedHandlers );
                    }
                    this._isAutomaticRequeryDisabled = value;
                }
            }
        }

        /// <summary>
        /// Method to determine if the command can be executed
        /// </summary>
        public bool CanExecute()
        {
            if( this._canExecuteMethod != null )
            {
                return this._canExecuteMethod();
            }
            return true;
        }

        /// <summary>
        /// Execution of the command
        /// </summary>
        public void Execute()
        {
            if( this._executeMethod != null )
            {
                this._executeMethod();
            }
        }

        /// <summary>
        /// Raises the <c>CanExecuteChaged</c> event
        /// </summary>
        public void RaiseCanExecuteChanged()
        {
            this.OnCanExecuteChanged();
        }

        /// <summary>
        /// Protected virtual method to raise <c>CanExecuteChanged</c> event
        /// </summary>
        protected virtual void OnCanExecuteChanged()
        {
            CommandManagerHelper.CallWeakReferenceHandlers( this._canExecuteChangedHandlers );
        }

        #endregion

        #region ICommand Members

        /// <summary>
        /// <see cref="ICommand"/>.CanExecuteChanged implementation
        /// </summary>
        public event EventHandler CanExecuteChanged
        {
            add
            {
                if( !this._isAutomaticRequeryDisabled )
                {
                    CommandManager.RequerySuggested += value;
                }
                CommandManagerHelper.AddWeakReferenceHandler( ref this._canExecuteChangedHandlers, value, 2 );
            }
            remove
            {
                if( !this._isAutomaticRequeryDisabled )
                {
                    CommandManager.RequerySuggested -= value;
                }
                CommandManagerHelper.RemoveWeakReferenceHandler( this._canExecuteChangedHandlers, value );
            }
        }

        bool ICommand.CanExecute( object parameter )
        {
            return this.CanExecute();
        }

        void ICommand.Execute( object parameter )
        {
            this.Execute();
        }

        #endregion

        #region Data

        private readonly Func<bool> _canExecuteMethod;

        private readonly Action _executeMethod;

        private List<WeakReference> _canExecuteChangedHandlers;

        private bool _isAutomaticRequeryDisabled;

        #endregion
    }


    /// <summary>
    /// This class contains methods for the <see cref="CommandManager"/> that help avoid memory leaks by
    /// using weak references.
    /// </summary>
    internal class CommandManagerHelper
    {
        internal static void CallWeakReferenceHandlers( List<WeakReference> handlers )
        {
            if( handlers != null )
            {
                // Take a snapshot of the handlers before we call out to them since the handlers
                // could cause the array to me modified while we are reading it.

                var callees = new EventHandler[ handlers.Count ];
                int count = 0;

                for( int i = handlers.Count - 1; i >= 0; i-- )
                {
                    WeakReference reference = handlers[ i ];
                    var handler = reference.Target as EventHandler;
                    if( handler == null )
                    {
                        // Clean up old handlers that have been collected
                        handlers.RemoveAt( i );
                    }
                    else
                    {
                        callees[ count ] = handler;
                        count++;
                    }
                }

                // Call the handlers that we snapshotted
                for( int i = 0; i < count; i++ )
                {
                    EventHandler handler = callees[ i ];
                    handler( null, EventArgs.Empty );
                }
            }
        }

        internal static void AddHandlersToRequerySuggested( List<WeakReference> handlers )
        {
            if( handlers != null )
            {
                foreach( WeakReference handlerRef in handlers )
                {
                    var handler = handlerRef.Target as EventHandler;
                    if( handler != null )
                    {
                        CommandManager.RequerySuggested += handler;
                    }
                }
            }
        }

        internal static void RemoveHandlersFromRequerySuggested( List<WeakReference> handlers )
        {
            if( handlers != null )
            {
                foreach( WeakReference handlerRef in handlers )
                {
                    var handler = handlerRef.Target as EventHandler;
                    if( handler != null )
                    {
                        CommandManager.RequerySuggested -= handler;
                    }
                }
            }
        }

        internal static void AddWeakReferenceHandler( ref List<WeakReference> handlers, EventHandler handler )
        {
            AddWeakReferenceHandler( ref handlers, handler, -1 );
        }

        internal static void AddWeakReferenceHandler( ref List<WeakReference> handlers, EventHandler handler, int defaultListSize )
        {
            if( handlers == null )
            {
                handlers = (defaultListSize > 0 ? new List<WeakReference>( defaultListSize ) : new List<WeakReference>());
            }

            handlers.Add( new WeakReference( handler ) );
        }

        internal static void RemoveWeakReferenceHandler( List<WeakReference> handlers, EventHandler handler )
        {
            if( handlers != null )
            {
                for( int i = handlers.Count - 1; i >= 0; i-- )
                {
                    WeakReference reference = handlers[ i ];
                    var existingHandler = reference.Target as EventHandler;
                    if( (existingHandler == null) || (existingHandler == handler) )
                    {
                        // Clean up old handlers that have been collected
                        // in addition to the handler that is to be removed.
                        handlers.RemoveAt( i );
                    }
                }
            }
        }
    }

    public class DelegateCommand<T> : ICommand
    {
        #region Constructors

        /// <summary>
        /// Initializes a new instance of the <see cref="DelegateCommand"/> class.
        /// </summary>
        /// <param name="executeMethod">The execute method.</param>
        /// <param name="parameter">The parameter.</param>
        public DelegateCommand( Action<T> executeMethod, T parameter )
            : this( executeMethod, null, false, parameter ) { }

        /// <summary>
        /// Initializes a new instance of the <see cref="DelegateCommand"/> class.
        /// </summary>
        /// <param name="executeMethod">The execute method.</param>
        /// <param name="canExecuteMethod">The can execute method.</param>
        /// <param name="parameter">The parameter.</param>
        public DelegateCommand( Action<T> executeMethod, Func<bool> canExecuteMethod, T parameter )
            : this( executeMethod, canExecuteMethod, false, parameter ) { }


        /// <summary>
        /// Initializes a new instance of the <see cref="DelegateCommand"/> class.
        /// </summary>
        /// <param name="executeMethod">The execute method.</param>
        /// <param name="canExecuteMethod">The can execute method.</param>
        /// <param name="isAutomaticRequeryDisabled">if set to <c>true</c> [is automatic requery disabled].</param>
        /// <param name="parameter">The parameter.</param>
        /// <exception cref="ArgumentNullException"><c>executeMethod</c> is null.</exception>
        public DelegateCommand( Action<T> executeMethod, Func<bool> canExecuteMethod, bool isAutomaticRequeryDisabled, T parameter )
        {
            if( executeMethod == null )
            {
                throw new ArgumentNullException( "executeMethod" );
            }

            this._executeMethod = executeMethod;
            this._parameter = parameter;
            this._canExecuteMethod = canExecuteMethod;
            this._isAutomaticRequeryDisabled = isAutomaticRequeryDisabled;
        }

        #endregion

        #region Public Methods

        /// <summary>
        /// Property to enable or disable CommandManager's automatic requery on this command
        /// </summary>
        public bool IsAutomaticRequeryDisabled
        {
            get { return this._isAutomaticRequeryDisabled; }
            set
            {
                if( this._isAutomaticRequeryDisabled != value )
                {
                    if( value )
                    {
                        CommandManagerHelper.RemoveHandlersFromRequerySuggested( this._canExecuteChangedHandlers );
                    }
                    else
                    {
                        CommandManagerHelper.AddHandlersToRequerySuggested( this._canExecuteChangedHandlers );
                    }
                    this._isAutomaticRequeryDisabled = value;
                }
            }
        }

        /// <summary>
        /// Method to determine if the command can be executed
        /// </summary>
        public bool CanExecute()
        {
            if( this._canExecuteMethod != null )
            {
                return this._canExecuteMethod();
            }
            return true;
        }

        /// <summary>
        /// Execution of the command
        /// </summary>
        public void Execute()
        {
            if( this._executeMethod != null )
            {
                this._executeMethod( this._parameter );
            }
        }

        /// <summary>
        /// Raises the <c>CanExecuteChaged</c> event
        /// </summary>
        public void RaiseCanExecuteChanged()
        {
            this.OnCanExecuteChanged();
        }

        /// <summary>
        /// Protected virtual method to raise <c>CanExecuteChanged</c> event
        /// </summary>
        protected virtual void OnCanExecuteChanged()
        {
            CommandManagerHelper.CallWeakReferenceHandlers( this._canExecuteChangedHandlers );
        }

        #endregion

        #region ICommand Members

        /// <summary>
        /// <see cref="ICommand"/>.CanExecuteChanged implementation
        /// </summary>
        public event EventHandler CanExecuteChanged
        {
            add
            {
                if( !this._isAutomaticRequeryDisabled )
                {
                    CommandManager.RequerySuggested += value;
                }
                CommandManagerHelper.AddWeakReferenceHandler( ref this._canExecuteChangedHandlers, value, 2 );
            }
            remove
            {
                if( !this._isAutomaticRequeryDisabled )
                {
                    CommandManager.RequerySuggested -= value;
                }
                CommandManagerHelper.RemoveWeakReferenceHandler( this._canExecuteChangedHandlers, value );
            }
        }

        bool ICommand.CanExecute( object parameter )
        {
            return this.CanExecute();
        }

        void ICommand.Execute( object parameter )
        {
            this.Execute();
        }

        #endregion

        #region Data

        private readonly Func<bool> _canExecuteMethod;

        private readonly Action<T> _executeMethod;

        private List<WeakReference> _canExecuteChangedHandlers;

        private bool _isAutomaticRequeryDisabled;

        private T _parameter;

        #endregion
    }
}
