var EventEmitter = require('events').EventEmitter;
var mongoose = require('mongoose');
var mongooseTimestamps = require('mongoose-timestamp');
var util = require('util');

function Database(config) {
    this.mongoose = mongoose.connect(config.databaseUrl, {
        server: {
            connectTimeoutMS: 30000,
            socketTimeoutMS: 30000
        },
        replset: {
            connectTimeoutMS: 30000,
            socketTimeoutMS: 30000
        }
    });
    this.db = mongoose.connection;
    this.createIndexes = config.createIndexes;

    this._schemas = {};

    this.db.on('error', console.error.bind(console, 'connection error:'));
    this.db.once('open', function callback() {
        console.log('Connected to MongoDB');
    });

    this.setupSchemas();
    this.setupModels();
}

util.inherits(Database, EventEmitter);

Database.prototype.ensureObjectId = function(id) {
    /*eslint-disable */
    return /^[0-9a-fA-F]{24}$/.test(id) ? mongoose.Types.ObjectId(id) : mongoose.Types.ObjectId('000000000000000000000000');
    /*eslint-enable */
};

Database.prototype.setupSchemas = function() {
    this._schemas.User = new mongoose.Schema({
        name: { type: String, required: true, maxlength: 100 },
        email: { type: String, required: true, lowercase: true, maxlength: 100 },
        password: { type: String, required: true }
    }, { autoIndex: this.createIndexes });
    this._schemas.User.index({ email: -1 }, { unique: true });
    this._schemas.User.plugin(mongooseTimestamps);
    this._schemas.User.set('toJSON', {
        transform: function(doc, ret) {
            delete ret.__v;
            return ret;
        }
    });

    this._schemas.Device = new mongoose.Schema({
        hardwareId: { type: String, required: true, index: true },
        accessToken: { type: String, required: true },
        user: { type: mongoose.Schema.Types.ObjectId, ref: 'User', required: true },
        name: { type: String, required: true, maxlength: 100 },
        type: { type: String, enum: ['wall_switch', 'wall_outlet', 'door_lock', 'hvac'], required: true },
        settings: {
            wall_switch: {
                enabled: { type: Boolean, default: false }
            },
            wall_outlet: {
                enabled: { type: Boolean, default: false },
                enabled2: { type: Boolean, default: false }
            },
            door_lock: {
                enabled: { type: Boolean, default: false }
            },
            hvac: {
                enabled: { type: Boolean, default: false },
                tempCurrent: { type: Number, default: 0 },
                tempDesired: { type: Number, default: 75 },
                mode: { type: String, enum: ['heat', 'cool'], default: 'cool' },
                fan: { type: String, enum: ['on', 'off', 'auto'], default: 'auto' }
            }
        }
    }, { autoIndex: this.createIndexes });
    this._schemas.Device.plugin(mongooseTimestamps);
    this._schemas.Device.set('toJSON', {
        transform: function(doc, ret) {
            if (ret.type && ret.settings) {
                ret.settings = ret.settings[ret.type];
            }
            delete ret.__v;
            return ret;
        }
    });
    this._schemas.Device.pre('save', function(next) {
        this.wasNew = this.isNew;
        next();
    });
    this._schemas.Device.post('save', function(doc) {
        if (doc.wasNew) {
            this.emit('deviceAuthenticate', doc);
        }

        this.emit('deviceUpdate', doc);
    }.bind(this));

    this._schemas.Task = new mongoose.Schema({
        user: { type: mongoose.Schema.Types.ObjectId, ref: 'User', required: true },
        name: { type: String, required: true, maxlength: 100 },
        actions: {
            type: [
                {
                    device: { type: mongoose.Schema.Types.ObjectId, ref: 'Device', required: true },
                    settings: {}
                }
            ],
            default: []
        }
    }, { autoIndex: this.createIndexes });
    this._schemas.Task.plugin(mongooseTimestamps);
    this._schemas.Task.set('toJSON', {
        transform: function(doc, ret) {
            delete ret.__v;
            return ret;
        }
    });

    this._schemas.Schedule = new mongoose.Schema({
        user: { type: mongoose.Schema.Types.ObjectId, ref: 'User', required: true },
        name: { type: String, required: true, maxlength: 100 },
        task: { type: mongoose.Schema.Types.ObjectId, ref: 'Task', required: true },
        interval: { type: String, required: true },
        nextRunAt: { type: Date, default: function() { return 0; }, required: true }
    }, { autoIndex: this.createIndexes });
    this._schemas.Schedule.plugin(mongooseTimestamps);
    this._schemas.Schedule.set('toJSON', {
        transform: function(doc, ret) {
            delete ret.__v;
            return ret;
        }
    });

    this._schemas.DeviceStat = new mongoose.Schema({
        user: { type: mongoose.Schema.Types.ObjectId, ref: 'User', required: true },
        device: { type: mongoose.Schema.Types.ObjectId, ref: 'Device', required: true },
        type: { type: String, enum: ['power_usage'], required: true },
        minute: {
            type: (function() {
                var d = {};
                for (var i = 0; i < 1440; i++) {
                    d[i] = Number;
                }
                return d;
            })(),
            default: function() {
                var d = {};
                for (var i = 0; i < 1440; i++) {
                    d[i] = 0;
                }
                return d;
            }
        }
    }, { autoIndex: this.createIndexes });
    this._schemas.DeviceStat.plugin(mongooseTimestamps);
    this._schemas.DeviceStat.set('toJSON', {
        virtuals: true,
        transform: function(doc, ret) {
            delete ret.__v;
            delete ret.minute;
            return ret;
        }
    });
    this._schemas.DeviceStat.virtual('hourly').get(function() {
        var hours = {};
        var peak, minutesUsed;

        for (var i = 0; i < 24; i++) {
            hours[i] = 0;
            peak = 0;
            minutesUsed = 0;

            // peak_k_watts / (60 / mins_used) = k_watts_hr
            for (var j = (i * 60); j < ((i + 1) * 60); j++) {
                if (peak < this.minute[j]) peak = this.minute[j];
                if (this.minute[j] > 0.01) minutesUsed += 1;
            }

            hours[i] = peak / (60 / minutesUsed);
        }

        return hours;
    });
    this._schemas.DeviceStat.virtual('daily').get(function() {
        var total = 0;

        for (var i = 0; i < 24; i++) {
            total += this.hourly[i];
        }

        return total;
    });

    this._schemas.Goal = new mongoose.Schema({
        user: { type: mongoose.Schema.Types.ObjectId, ref: 'User', required: true },
        type: { type: String, enum: ['power_usage_daily', 'power_usage_weekly', 'power_usage_monthly', 'power_usage_yearly'], required: true },
        device: { type: mongoose.Schema.Types.ObjectId, ref: 'Device' },
        limit: Number
    }, { autoIndex: this.createIndexes });
    this._schemas.Goal.plugin(mongooseTimestamps);
    this._schemas.Goal.set('toJSON', {
        transform: function(doc, ret) {
            delete ret.__v;
            return ret;
        }
    });
};

Database.prototype.setupModels = function() {
    Object.keys(this._schemas).forEach(function(schema) {
        this[schema] = mongoose.model(schema, this._schemas[schema]);
    }.bind(this));
};

module.exports = Database;
