var crypto = require('crypto');
var router = require('express').Router(); //eslint-disable-line
var helpers = require('../helpers/devices');
var async = require('async');
var moment = require('moment');

module.exports = function(app, db) {
    router.get(
        '/',
        app.auth,
        function(req, res) {
            db.Device.find({
                user: req.user._id
            })
            .sort({
                _id: 1
            })
            .exec(function(err, devices) {
                if (err) {
                    return res.status(500).json({
                        message: 'internal server error'
                    });
                }

                var startOfDay = moment().startOf('day').toDate();
                var currentMinute = moment().diff(startOfDay, 'minutes');

                var finalDevices = [];

                async.eachSeries(devices, function(d, callback) {
                    d = d.toJSON();

                    db.DeviceStat.findOne({
                        user: req.user._id,
                        device: d._id,
                        createdAt: {
                            $gte: startOfDay
                        }
                    }, function(energyErr, device) {
                        if (energyErr) return callback(err);

                        if (!device) {
                            d.currentEnergy = 0;
                        } else {
                            d.currentEnergy = device.minute[currentMinute - 1] || device.minute[currentMinute] || 0;
                        }

                        finalDevices.push(d);

                        callback();
                    });
                }, function(e) {
                    if (e) {
                        return res.status(500).json({
                            message: 'internal server error'
                        });
                    }

                    res.json(finalDevices);
                });
            });
        }
    );

    router.post(
        '/',
        app.auth,
        function(req, res) {
            new db.Device({
                hardwareId: req.body.hardwareId,
                accessToken: crypto.randomBytes(20).toString('hex'),
                user: req.user._id,
                name: req.body.name,
                type: helpers.parseTypeFromHardwareId(req.body.hardwareId)
            }).save(function(err, saved) {
                if (err) {
                    return res.status(400).json({
                        message: err.message
                    });
                }

                res.json(saved);
            });
        }
    );

    router.get(
        '/:id',
        app.auth,
        function(req, res) {
            db.Device.findOne({
                _id: db.ensureObjectId(req.params.id),
                user: req.user._id
            }, function(err, device) {
                if (err) {
                    return res.status(500).json({
                        message: 'internal server error'
                    });
                }

                if (!device) {
                    return res.status(404).json({
                        message: 'device does not exist'
                    });
                }

                res.json(device);
            });
        }
    );

    router.put(
        '/:id',
        app.auth,
        function(req, res) {
            db.Device.findOne({
                _id: db.ensureObjectId(req.params.id),
                user: req.user._id
            }, function(err, device) {
                if (err) {
                    return res.status(500).json({
                        message: 'internal server error'
                    });
                }

                if (!device) {
                    return res.status(404).json({
                        message: 'device does not exist'
                    });
                }

                var invalidSettings = helpers.isInvalidDeviceSettings(db.Device, device.type, req.body.settings);
                if (invalidSettings) {
                    return res.status(400).json({
                        message: invalidSettings.message
                    });
                }

                device.name = req.body.name;

                device.settings[device.type] = req.body.settings;

                device.save(function(saveErr, saved) {
                    if (saveErr) {
                        return res.status(400).json({
                            message: err.message
                        });
                    }

                    res.json(saved);
                });
            });
        }
    );

    router.delete(
        '/:id',
        app.auth,
        function(req, res) {
            db.Device.findOne({
                _id: db.ensureObjectId(req.params.id),
                user: req.user._id
            }, function(err, device) {
                if (err) {
                    return res.status(500).json({
                        message: 'internal server error'
                    });
                }

                if (!device) {
                    return res.status(404).json({
                        message: 'device does not exist'
                    });
                }

                device.remove(function(saveErr) {
                    if (saveErr) {
                        return res.status(500).json({
                            message: 'internal server error'
                        });
                    }

                    res.status(204).send();
                });
            });
        }
    );

    return router;
};
