var async = require('async');
var moment = require('moment');
var router = require('express').Router(); //eslint-disable-line

module.exports = function(app, db) {
    router.get(
        '/',
        app.auth,
        function(req, res) {
            db.DeviceStat.find({
                user: req.user._id,
                createdAt: {
                    $gte: moment().startOf('year').toISOString()
                }
            }, function(err, stats) {
                if (err) {
                    return res.status(500).json({
                        message: 'internal server error'
                    });
                }

                res.json(stats);
            });
        }
    );

    router.get(
        '/goals',
        app.auth,
        function(req, res) {
            db.Goal.find({
                user: req.user._id
            }, function(err, goals) {
                if (err) {
                    return res.status(500).json({
                        message: 'internal server error'
                    });
                }

                var _goals = [];

                async.each(goals, function(goal, callback) {
                    var q = {
                        user: req.user._id
                    };

                    if (goal.device) {
                        q.device = goal.device;
                    }

                    switch (goal.type) {
                        case 'power_usage_daily':
                            q.createdAt = {
                                $gte: moment().startOf('day').toDate()
                            };
                            break;
                        case 'power_usage_weekly':
                            q.createdAt = {
                                $gte: moment().startOf('week').toDate()
                            };
                            break;
                        case 'power_usage_monthly':
                            q.createdAt = {
                                $gte: moment().startOf('month').toDate()
                            };
                            break;
                        case 'power_usage_yearly':
                            q.createdAt = {
                                $gte: moment().startOf('year').toDate()
                            };
                            break;
                        default:
                            callback(new Error('invalid goal type'));
                            return;
                    }

                    db.DeviceStat.find(q).exec(function(dbErr, stats) {
                        if (dbErr) {
                            return callback(err);
                        }

                        var total = 0;

                        stats.forEach(function(stat) {
                            total += stat.daily;
                        });

                        goal = goal.toObject();

                        goal.usage = total;

                        _goals.push(goal);

                        callback();
                    });
                }, function(err) {
                    if (err) {
                        return res.status(500).json({
                            message: 'internal server error'
                        });
                    }

                    res.json(_goals);
                });
            });
        }
    );

    router.post(
        '/goals',
        app.auth,
        function(req, res) {
            new db.Goal({
                user: req.user._id,
                limit: req.body.limit,
                device: req.body.device,
                type: req.body.type
            }).save(function(err, saved) {
                if (err) {
                    return res.status(500).json({
                        message: 'internal server error'
                    });
                }

                res.json(saved);
            });
        }
    );

    router.put(
        '/goals/:id',
        app.auth,
        function(req, res) {
            db.Goal.findOne({
                _id: db.ensureObjectId(req.params.id),
                user: req.user._id
            }, function(err, goal) {
                if (err) {
                    return res.status(500).json({
                        message: 'internal server error'
                    });
                }

                if (!goal) {
                    return res.status(404).json({
                        message: 'goal does not exist'
                    });
                }

                goal.type = req.body.type;
                goal.limit = req.body.limit;
                goal.device = req.body.device;

                goal.save(function(saveErr, saved) {
                    if (saveErr) {
                        return res.status(500).json({
                            message: 'internal server error'
                        });
                    }

                    res.json(saved);
                });
            });
        }
    );

    router.delete(
        '/goals/:id',
        app.auth,
        function(req, res) {
            db.Goal.findOne({
                _id: db.ensureObjectId(req.params.id),
                user: req.user._id
            }, function(err, goal) {
                if (err) {
                    return res.status(500).json({
                        message: 'internal server error'
                    });
                }

                if (!goal) {
                    return res.status(404).json({
                        message: 'goal does not exist'
                    });
                }

                goal.remove(function(saveErr) {
                    if (saveErr) {
                        return res.status(500).json({
                            message: 'internal server error'
                        });
                    }

                    res.status(204).send();
                });
            });
        }
    );

    return router;
};
