var parser = require('cron-parser');
var async = require('async');
var router = require('express').Router(); //eslint-disable-line

var cronValidator = function(interval) {
    if (!interval) {
        return new Error('Invalid interval');
    }

    try {
        var intervalParts = interval.split(' ');

        // Tests parsing (throws on error)
        interval = parser.parseExpression(interval);

        // cron-parser allows seconds, so we must prevent them.
        if (intervalParts.length > 5) throw new Error('Invalid Cron Format');

        if (
            (intervalParts[0].indexOf('/') > -1 && parseInt(intervalParts[0].split('/')[1], 10) < 5) ||
            intervalParts[0].indexOf('-') > -1
        ) {
            throw new Error('Smallest interval is once per 5 minutes');
        }

        if (intervalParts[0].indexOf(',') > -1) {
            var intervals = intervalParts[0].split(',');

            intervals.map(function(val) {
                return parseInt(val, 10);
            });

            intervals.sort(function(a, b) {
                return a - b;
            });

            for (var i = 0; i < intervals.length - 1; i++) {
                if (intervals[i + 1] - intervals[i] >= 5) continue;

                throw new Error('Smallest interval is once per 5 minutes');
            }
        }
    } catch (e) {
        return e;
    }

    return null;
};

module.exports = function(app, db) {
    setInterval(function() {
        db.Schedule.find({
            nextRunAt: {
                $lte: new Date()
            }
        })
        .populate('task')
        .populate('task.actions.device')
        .exec(function(err, events) {
            async.each(events, function(event, callback) {
                if(!event.task || !event.task.actions || !event.task.actions.length) {
                    return callback();
                }

                db.Task.populate(event.task, 'actions.device', function(err) {
                    if (err) {
                        return callback();
                    }

                    async.eachSeries(event.task.actions, function(action, cb) {
                        action.device.settings[action.device.type] = action.settings;

                        action.device.save(cb);
                    }, function() {
                        event.nextRunAt = parser.parseExpression(event.interval).next();

                        event.save(callback);
                    });
                });
            }, function(schErr) {
                if (schErr) console.log(err);
            });
        });
    }, 60000);

    router.get(
        '/',
        app.auth,
        function(req, res) {
            db.Schedule.find({
                user: req.user._id
            }, function(err, schedules) {
                if (err) {
                    return res.status(500).json({
                        message: 'internal server error'
                    });
                }

                res.json(schedules);
            });
        }
    );

    router.post(
        '/',
        app.auth,
        function(req, res) {
            var intervalError = cronValidator(req.body.interval);

            if (intervalError) {
                res.status(400).json({
                    message: intervalError.message
                });
                return;
            }

            new db.Schedule({
                user: req.user._id,
                task: req.body.task,
                name: req.body.name,
                interval: req.body.interval,
                nextRunAt: parser.parseExpression(req.body.interval).next()
            }).save(function(err, saved) {
                if (err) {
                    return res.status(500).json({
                        message: 'internal server error'
                    });
                }

                res.json(saved);
            });
        }
    );

    router.get(
        '/:id',
        app.auth,
        function(req, res) {
            db.Schedule.findOne({
                _id: db.ensureObjectId(req.params.id),
                user: req.user._id
            }, function(err, schedule) {
                if (err) {
                    return res.status(500).json({
                        message: 'internal server error'
                    });
                }

                if (!schedule) {
                    return res.status(404).json({
                        message: 'schedule does not exist'
                    });
                }

                res.json(schedule);
            });
        }
    );

    router.put(
        '/:id',
        app.auth,
        function(req, res) {
            db.Schedule.findOne({
                _id: db.ensureObjectId(req.params.id),
                user: req.user._id
            }, function(err, schedule) {
                if (err) {
                    return res.status(500).json({
                        message: 'internal server error'
                    });
                }

                if (!schedule) {
                    return res.status(404).json({
                        message: 'schedule does not exist'
                    });
                }

                var intervalError = cronValidator(req.body.interval);

                if (intervalError) {
                    res.status(400).json({
                        message: intervalError.message
                    });
                    return;
                }

                schedule.name = req.body.name;
                schedule.task = req.body.task;
                schedule.interval = req.body.interval;
                schedule.nextRunAt = parser.parseExpression(req.body.interval).next();

                schedule.save(function(saveErr, saved) {
                    if (saveErr) {
                        return res.status(500).json({
                            message: 'internal server error'
                        });
                    }

                    res.json(saved);
                });
            });
        }
    );

    router.delete(
        '/:id',
        app.auth,
        function(req, res) {
            db.Schedule.findOne({
                _id: db.ensureObjectId(req.params.id),
                user: req.user._id
            }, function(err, schedule) {
                if (err) {
                    return res.status(500).json({
                        message: 'internal server error'
                    });
                }

                if (!schedule) {
                    return res.status(404).json({
                        message: 'schedule does not exist'
                    });
                }

                schedule.remove(function(saveErr) {
                    if (saveErr) {
                        return res.status(500).json({
                            message: 'internal server error'
                        });
                    }

                    res.status(204).send();
                });
            });
        }
    );

    return router;
};
