var async = require('async');
var router = require('express').Router(); //eslint-disable-line
var helpers = require('../helpers/devices');

module.exports = function(app, db) {
    router.get(
        '/',
        app.auth,
        function(req, res) {
            db.Task.find({
                user: req.user._id
            }, function(err, tasks) {
                if (err) {
                    return res.status(500).json({
                        message: 'internal server error'
                    });
                }

                res.json(tasks);
            });
        }
    );

    router.post(
        '/',
        app.auth,
        function(req, res) {
            req.body.actions = req.body.actions || [];

            if (!Array.isArray(req.body.actions)) {
                return res.status(400).json({
                    message: 'invalid actions'
                });
            }

            async.each(req.body.actions, function(device, callback) {
                db.Device.findOne({
                    _id: db.ensureObjectId(device.device),
                    user: req.user._id
                }, function(err, d) {
                    if (err) {
                        return callback(new Error('internal server error'));
                    }

                    if (!d) {
                        return callback(new Error('device not found'));
                    }

                    var invalidSettings = helpers.isInvalidDeviceSettings(db.Device, d.type, device.settings);
                    if (invalidSettings) {
                        return callback(invalidSettings);
                    }

                    callback();
                });
            }, function(err) {
                if (err) {
                    return res.status(400).json({
                        message: err.message
                    });
                }

                new db.Task({
                    user: req.user._id,
                    name: req.body.name,
                    actions: req.body.actions
                }).save(function(saveErr, saved) {
                    if (saveErr) {
                        return res.status(400).json({
                            message: err.message
                        });
                    }

                    res.json(saved);
                });
            });
        }
    );

    router.get(
        '/:id',
        app.auth,
        function(req, res) {
            db.Task.findOne({
                _id: db.ensureObjectId(req.params.id),
                user: req.user._id
            }, function(err, task) {
                if (err) {
                    return res.status(500).json({
                        message: 'internal server error'
                    });
                }

                if (!task) {
                    return res.status(404).json({
                        message: 'task does not exist'
                    });
                }

                res.json(task);
            });
        }
    );

    router.put(
        '/:id',
        app.auth,
        function(req, res) {
            db.Task.findOne({
                _id: db.ensureObjectId(req.params.id),
                user: req.user._id
            }, function(err, task) {
                if (err) {
                    return res.status(500).json({
                        message: 'internal server error'
                    });
                }

                if (!task) {
                    return res.status(404).json({
                        message: 'task does not exist'
                    });
                }

                req.body.actions = req.body.actions || [];

                if (!Array.isArray(req.body.actions)) {
                    return res.status(400).json({
                        message: 'invalid actions'
                    });
                }

                async.each(req.body.actions, function(action, callback) {
                    db.Device.findOne({
                        _id: db.ensureObjectId(action.device),
                        user: req.user._id
                    }, function(devErr, d) {
                        if (devErr) {
                            return callback(new Error('internal server error'));
                        }

                        if (!d) {
                            return callback(new Error('device not found'));
                        }

                        var invalidSettings = helpers.isInvalidDeviceSettings(db.Device, d.type, action.settings);
                        if (invalidSettings) {
                            return callback(invalidSettings);
                        }

                        callback();
                    });
                }, function(loopErr) {
                    if (loopErr) {
                        return res.status(400).json({
                            message: err.message
                        });
                    }

                    task.actions = req.body.actions;
                    task.name = req.body.name;

                    task.save(function(saveErr, saved) {
                        if (saveErr) {
                            return res.status(400).json({
                                message: err.message
                            });
                        }

                        res.json(saved);
                    });
                });
            });
        }
    );

    router.post(
        '/:id/run',
        app.auth,
        function(req, res) {
            db.Task.findOne({
                _id: db.ensureObjectId(req.params.id),
                user: req.user._id
            })
            .populate('actions.device')
            .exec(function(err, task) {
                if (err) {
                    return res.status(500).json({
                        message: 'internal server error'
                    });
                }

                if (!task) {
                    return res.status(404).json({
                        message: 'task does not exist'
                    });
                }

                async.each(task.actions, function(action, callback) {
                    action.device.settings[action.device.type] = action.settings;

                    action.device.save(callback);
                }, function(execErr) {
                    if (execErr) {
                        return res.status(500).json({
                            message: 'internal server error'
                        });
                    }

                    res.status(204).send();
                });
            });
        }
    );

    router.delete(
        '/:id',
        app.auth,
        function(req, res) {
            db.Task.findOne({
                _id: db.ensureObjectId(req.params.id),
                user: req.user._id
            }, function(err, task) {
                if (err) {
                    return res.status(500).json({
                        message: 'internal server error'
                    });
                }

                if (!task) {
                    return res.status(404).json({
                        message: 'task does not exist'
                    });
                }

                task.remove(function(saveErr) {
                    if (saveErr) {
                        return res.status(500).json({
                            message: 'internal server error'
                        });
                    }

                    res.status(204).send();
                });
            });
        }
    );

    return router;
};
