var bcrypt = require('bcrypt');
var crypto = require('crypto');
var validator = require('validator');
var router = require('express').Router(); //eslint-disable-line

module.exports = function(app, db) {
    router.get(
        '/',
        app.auth,
        function(req, res) {
            res.json(req.user);
        }
    );

    router.post(
        '/',
        function(req, res, next) {
            // Validate action param
            if (!req.body.action || ['register', 'login'].indexOf(req.body.action) === -1) {
                return res.status(400).json({
                    message: 'invalid action specified'
                });
            }

            // Passwords and emails are required
            if (!req.body.password || !req.body.email) {
                return res.status(400).json({
                    message: 'email and password required'
                });
            }

            // Validate email
            if (!validator.isEmail(req.body.email)) {
                return res.status(400).json({
                    message: 'valid email required'
                });
            }

            // Handle authentications
            if (req.body.action === 'login') {
                db.User.findOne({
                    email: req.body.email
                }, function(err, user) {
                    if (err) {
                        return res.status(500).json({
                            message: 'internal server error'
                        });
                    }

                    // Always compute password validation (prevents people querying if a email is registered)
                    bcrypt.compare(req.body.password, (user && user.password) || '$2a$08$TWAnI8oSI0nmXw61LZzLIe1KT1hatDS5juD21ToNXroX3BjIWwFd6', function(hashErr, verified) {
                        if (hashErr) {
                            return res.status(500).json({
                                message: 'error validating password'
                            });
                        }

                        if (!user || !verified) {
                            return res.status(400).json({
                                message: 'email or password invalid'
                            });
                        }

                        req.user = user;
                        next();
                    });
                });
                return;
            }

            // Otherwise handles registrations

            // Require name
            if (!req.body.name) {
                return res.status(400).json({
                    message: 'name required'
                });
            }

            // Validate password
            if (req.body.passwordConfirm !== req.body.password) {
                return res.status(400).json({
                    message: 'passwords do not match'
                });
            }

            // Compute hash and create user account
            bcrypt.hash(req.body.password, 8, function(err, hash) {
                new db.User({
                    email: req.body.email,
                    name: req.body.name,
                    password: hash
                }).save(function(saveErr, user) {
                    if (saveErr || !user) {
                        return res.status(400).json({
                            message: 'email already registered'
                        });
                    }

                    req.user = user;
                    next();
                });
            });
        },
        function(req, res) {
            req.session._id = req.user._id;
            req.session.name = req.user.name;
            req.session.email = req.user.email;
            req.session.password = req.user.password;
            req.session.token = crypto.randomBytes(16).toString('hex');

            req.session.save(function(err) {
                if (err) {
                    return res.status(500).json({
                        message: 'internal server error'
                    });
                }

                var user = req.user.toObject();

                user.sessionToken = req.session.token;

                res.json(user);
            });
        }
    );

    router.put(
        '/',
        app.auth,
        function(req, res, next) {
            if (!req.body.email) {
                return res.status(400).json({
                    message: 'email is required'
                });
            }

            // Validate email
            if (!validator.isEmail(req.body.email)) {
                return res.status(400).json({
                    message: 'valid email required'
                });
            }

            // Replace email if changed
            if (req.body.email !== req.user.email) {
                req.user.email = req.body.email;
            }

            // Replace name if changed
            if (req.body.name !== req.user.name) {
                req.user.name = req.body.name;
            }

            // Validate password if changed
            if (req.body.passwordConfirm && req.body.passwordConfirm !== req.body.password) {
                return res.status(400).json({
                    message: 'passwords do not match'
                });
            }

            // Replace password if changed
            if (req.body.passwordConfirm) {
                bcrypt.hash(req.body.password, 8, function(err, hash) {
                    req.user.password = hash;
                    next();
                });
                return;
            }

            next();
        },
        function(req, res) {
            req.user.save(function(err, user) {
                if (err) {
                    return res.status(500).json({
                        message: 'internal server error'
                    });
                }

                res.json(user);
            });
        }
    );

    router.delete(
        '/',
        app.auth,
        function(req, res) {
            req.user.remove(function(err) {
                if (err) {
                    return res.status(500).json({
                        message: 'internal server error'
                    });
                }

                res.status(204).send();
            });
        }
    );

    return router;
};
