var net = require('net');
var moment = require('moment');

function SS(config, server, db) {
    if (!(this instanceof SS)) return new SS(config, server, db);

    this.db = db;
    this.clients = {};
    this.hardwareClients = {};

    this.ss = net.createServer(this.connection.bind(this));
    this.ss.listen(config.port, function() {
        console.log('Started ss on port:', config.port);
    });

    this.db.on('deviceAuthenticate', function(device) {
        this.emit(this.hardwareClients[device.hardwareId], 'authenticate', 'access_token', device.accessToken);
    }.bind(this));

    this.db.on('deviceUpdate', function(device) {
        device = device.toObject();
        var settings = device.settings[device.type];

        Object.keys(settings).forEach(function(name) {
            this.emit(device._id.toString(), 'set', name, settings[name].toString());
        }.bind(this));
    }.bind(this));

    // Keep the sockets alive by pinging them
    setInterval(function() {
        Object.keys(this.clients).forEach(function(clientId) {
            this.emit(clientId, 'keepalive', 'ping', Date.now());
        }.bind(this));
    }.bind(this), 30000);
}

// TODO: create methods for socket 'action's

// Handles incoming socket connections
SS.prototype.connection = function(socket) {
    var buffer = '';
    socket.authenticated = false;
    socket.deviceId = null;
    socket.hardwareId = null;
    socket.lastPong = Date.now();

    // Socket format:
    // "[device_id] [action] [state_name] [state_value(optional)]\n"
    socket.on('data', function(data) {
        // Append data to buffer
        buffer += data.toString();

        // Split buffer by new line char,
        // and loop through finished client commands
        var pieces = buffer.replace(/\r/g, '').split('\n');
        var params, hardwareId, action, stateName, stateValue;
        while (pieces.length > 1) {
            params = pieces.shift().split(' ');

            // Check if command is properly formatted
            if (params.length < 3) {
                this.emit(socket, 'error', 'invalid_params');
                continue;
            }

            hardwareId = params.shift();
            action = params.shift();
            stateName = params.shift();
            stateValue = params.join(' ');

            // Checks that action method exists
            if (!action || ['set', 'authenticate', 'keepalive', 'stat'].indexOf(action) === -1) {
                this.emit(socket, 'error', 'invalid_action');
                continue;
            }

            this.hardwareClients[hardwareId] = socket;

            // Disallows unauthed users from executing actions
            if (action !== 'authenticate' && !socket.authenticated) {
                this.emit(socket, 'error', 'unauthorized');
                continue;
            }

            action = this[action];

            // Runs action
            action.call(this, socket, hardwareId, stateName, stateValue);
        }

        // Restore unhandled buffer
        buffer = pieces[0];
    }.bind(this));

    socket.on('end', function() {
        // Handle case where client isn't authorized
        if (!socket.deviceId) return;

        // Remove from device clients list
        delete this.clients[socket.deviceId];
    }.bind(this));

    socket.on('error', function() {
        // Handle case where client isn't authorized
        if (!socket.deviceId) return;

        // Remove from device clients list
        delete this.clients[socket.deviceId];
    }.bind(this));
};

// Handles keepalive ping/pong
SS.prototype.keepalive = function(socket, hardwareId, stateName, stateValue) {
    if (stateName === 'ping') {
        this.emit(socket, 'keepalive', 'pong', stateValue);
        return;
    }

    socket.lastPong = Date.now();
};

// Handles setting of device settings
SS.prototype.set = function(socket, hardwareId, stateName, stateValue) {
    if (!stateValue) {
        return this.emit(socket, 'error', 'state_value_required');
    }

    this.db.Device.findOne({
        hardwareId: hardwareId
    }, function(err, device) {
        // Check if there was a database error
        if (err) {
            this.emit(socket, 'error', 'internal_server_error');
            return;
        }

        // Check if device exists and token provided is correct
        if (!device) {
            this.emit(socket, 'error', 'device_not_found');
            return;
        }

        if (!(stateName in device.settings[device.type])) {
            this.emit(socket, 'error', 'invalid_state_name');
            return;
        }

        if (device.settings[device.type][stateName].toString() === stateValue) {
            return;
        }

        device.settings[device.type][stateName] = stateValue;

        device.save(function(saveErr) {
            if (!saveErr) return;

            this.emit(socket, 'error', 'invalid_state_value');
        }.bind(this));
    }.bind(this));
};

// Handles setting of device stats
SS.prototype.stat = function(socket, hardwareId, stateName, stateValue) {
    if (!stateValue) {
        return this.emit(socket, 'error', 'state_value_required');
    }

    this.db.Device.findOne({
        hardwareId: hardwareId
    }, function(err, device) {
        // Check if there was a database error
        if (err) {
            this.emit(socket, 'error', 'internal_server_error');
            return;
        }

        // Check if device exists and token provided is correct
        if (!device) {
            this.emit(socket, 'error', 'device_not_found');
            return;
        }

        if (stateName !== 'power_usage') {
            this.emit(socket, 'error', 'invalid_state_name');
            return;
        }

        var startOfDay = moment().startOf('day').toDate();
        var currentMinute = moment().diff(startOfDay, 'minutes');

        this.db.DeviceStat.findOne({
            user: device.user,
            device: device._id,
            createdAt: startOfDay
        }, function(statErr, stat) {
            if (statErr) {
                this.emit(socket, 'error', 'internal_server_error');
            }

            if (!stat) {
                stat = new this.db.DeviceStat({
                    createdAt: startOfDay,
                    type: 'power_usage',
                    device: device._id,
                    user: device.user
                });
            }

            stat.minute[currentMinute] += stat.minute[currentMinute] + parseFloat(stateValue, 10) / 1000;
            stat.markModified('minute');

            stat.save(function(saveErr) {
                if (!saveErr) return;

                this.emit(socket, 'error', 'invalid_state_value');
            }.bind(this));
        }.bind(this));
    }.bind(this));
};

// Handles authentication of new sockets
SS.prototype.authenticate = function(socket, hardwareId, accessTokenState, accessToken) {
    if (accessTokenState !== 'access_token') {
        return this.emit(socket, 'error', 'invalid_state_name');
    }

    if (!accessToken) {
        return this.emit(socket, 'error', 'access_token_required');
    }

    this.db.Device.findOne({
        hardwareId: hardwareId,
        accessToken: accessToken
    }, function(err, device) {
        // Check if there was a database error
        if (err) {
            this.emit(socket, 'error', 'internal_server_error');
            return;
        }

        // Check if device exists and token provided is correct
        if (!device) {
            this.emit(socket, 'error', 'authentication_error');
            return;
        }

        // Authenticate device
        socket.hardwareId = device.hardwareId;
        socket.deviceId = device.id;
        socket.authenticated = true;

        // Associate socket to device clients list
        this.clients[device.id] = socket;

        // Emit successful authentication
        this.emit(socket, 'authenticate', 'success');

        // Push current stored settings to device
        this.db.emit('deviceUpdate', device);
    }.bind(this));
};

// Handles outgoing socket commands
SS.prototype.emit = function(socketOrDevice, action, stateName, stateValue) {
    // Resolves device IDs to sockets
    if (typeof socketOrDevice === 'string') {
        socketOrDevice = this.clients[socketOrDevice];
    }

    // If device not connected, bail out
    if (!socketOrDevice) return;

    // Compile params
    var params = [socketOrDevice.hardwareId || '000000', action, stateName];
    if (stateValue) params.push(stateValue);

    // Sent out to client
    socketOrDevice.write(params.join(' ') + '\n');
};

module.exports = SS;
