var bodyParser = require('body-parser');
var cors = require('cors');
var express = require('express');
var http = require('http');
var logger = require('morgan');
var methodOverride = require('method-override');
var session = require('express-session');
var MongoStore = require('connect-mongo')(session);

var config = require('./config.json');
var DB = require('./app/db');
var ss = require('./app/ss');

function KnightHome() {
    var db = new DB(config.mongo);

    // Instantiate routing
    var app = express();

    // Configure logger
    if (config.express.logLevel) app.use(logger(config.express.logLevel));

    // CORS allows cross-domain HTTP requests
    app.use(cors(function(req, callback) {
        // If request originates from origin, allow authenticated requests
        if ((new RegExp(config.express.origin + '$', 'i')).test(req.header('Origin'))) {
            callback(null, {
                origin: true,
                credentials: true,
                maxAge: 2592000
            });
        } else {
            callback(null, {
                origin: true,
                maxAge: 2592000
            });
        }
    }));

    // Allows bodies to be posted URL-encoded or JSON
    app.use(bodyParser.json());
    app.use(bodyParser.urlencoded({
        extended: true
    }));

    // Allows REST-ful HTTP methods to be used
    app.use(methodOverride());

    // Configures session cookies
    app.use(session({
        secret: config.express.secret,
        name: 'kh-session',
        resave: false,
        rolling: false,
        saveUninitialized: false,
        cookie: {
            maxAge: 1296000000
        },
        store: new MongoStore({
            mongooseConnection: db.db
        })
    }));

    // Removes express versioning header
    app.set('x-powered-by', false);

    // Resolves IP header forwarding
    app.set('trust proxy', function() { return true; });

    // Start listening on port
    var server = http.createServer(app);
    server.listen(config.express.port, function() {
        console.log('API listening on: ' + config.express.port);
    });

    // Start TLS socket server
    app.ss = ss(config.socketServer, server, db);

    // Assign API routes
    require('./app/routes')(app, db);
}

module.exports = new KnightHome();
