#include <msp430fr6922.h>

//Defines
#define FLAGS UCA1IFG // Contains the transmit & receive flags
#define RXFLAG UCRXIFG // Receive flag
#define TXFLAG UCTXIFG // Transmit flag
#define TXBUFFER UCA1TXBUF // Transmit buffer
#define RXBUFFER UCA1RXBUF // Receive buffer
#define redLED BIT1
#define greenLED BIT7

/*
 *
 * LEDs
 * 0 - PJ.0 - I2C Indicator
 * 1 - PJ.1
 * 2 - PJ.2
 * 3 - PJ.3
 * 4 - P3.3 - IR Indicator
 * 5 - P7.0 - Recording Indicator
 * 6 - P7.1 - "Power Indicator"
 *
 *LCD
 * VSS - GND
 * VDD - 5V
 * V0 - pot
 * RS - Register select - pin 39 - P7.2
 * R/W - Read / Write - GND
 * E - Enable - pin 40 - p7.3
 * D0 - pin 41 - P7.4
 * D1 - pin 44 - P1.3
 * D2 - pin 45 - P1.2
 * D3 - pin 46 - P1.1
 * D4 - pin 47 - P1.0
 * D5 - pin 48 - P9.4
 * D6 -  pin 49 - P9.5
 * D7 - pin 50 - P9.6
 * A - 5V
 * K - GND
 *
 *
 *IR
 *  OUT - P4.4
 *  Vs - 5V
 *  GND - GND
 *
 */

void clear_i2c(void){

    volatile unsigned int i;

    //set 6 and 7 to outputs
    P1DIR |= (BIT6|BIT7);
    //Divert Pin functionality to default (00)
    P1SEL1 &= ~(BIT6|BIT7);
    P1SEL0 &= ~(BIT6|BIT7);
    //Set outputs to 0
    P1OUT &= ~(BIT6|BIT7);


    //Toggle clock line 16 times (P1.7)
    for(i=0; i<32; i++){
        P1OUT ^= BIT7;
    }

}

void init_I2C(void) {

    UCB0CTLW0 |= UCSWRST;

    //Divert Pin functionality
    P1SEL1 &= ~(BIT6|BIT7);
    P1SEL0 |= (BIT6|BIT7);

    //Set to I2C mode, set as master, and select the SMCLK as the source clock
    UCB0CTLW0 |= UCMODE_3 | UCMST | UCSSEL_3;

    //Clock Divider = 8
    //Clock = ~131 KHz
    UCB0BRW = 8;

    UCB0CTLW0 &= ~UCSWRST;

}

int i2c_read_word(unsigned char i2c_address, unsigned int * data) {
    unsigned char byte1, byte2;
    // Initialize the bytes to make sure data is received every time

    //wait for bus to be clear
    while(UCB0STAT & UCBBUSY);
    //set i2c slave adddress to read from
    UCB0I2CSA = i2c_address;
    UCB0IFG &= ~(UCTXIFG0 & UCRXIFG0);
    //set to write mode and transmit start
    UCB0CTLW0 |= (UCTR | UCTXSTT);

    //WAIT FOR START SIGNAL TO FINISH
    while((UCB0CTLW0 & UCTXSTT) != 0);

    //SET TO READ MODE
    UCB0CTLW0 &= ~UCTR;
    //TRANSMIT START RECEIVING
    UCB0CTLW0 |= UCTXSTT;

    while(!(UCB0IFG & UCRXIFG0));
    //set byte 1 = rx buffer
    byte1 = UCB0RXBUF;


    //wait for start to transmit
//    while((UCB0CTLW0 & UCTXSTT) != 0);
    //send stop signal
    UCB0CTLW0 |= UCTXSTP;
    //wait for RX flag
    while(!(UCB0IFG & UCRXIFG0));
    //set byte 1 = rx buffer
    byte2 = UCB0RXBUF;

    while(UCB0CTLW0 & UCTXSTP);

    *data = byte2;



    return 0;
}

int i2c_write_word(unsigned char i2c_address, unsigned int data) {
    unsigned char byte1, byte2;
    byte1 = (data >> 8) & 0xFF; // MSByte
    byte2 = data & 0xFF; // LSByte

    PJOUT |= BIT0;

    UCB0I2CSA = i2c_address; // Set I2C address
    UCB0CTLW0 |= UCTR; // Master writes (R/W bit = Write)

    UCB0CTLW0 |= UCTXSTT; // Initiate the Start Signal
    while ((UCB0IFG & UCTXIFG0) ==0) {}

    //********** Write Byte #1 ***************************
    UCB0TXBUF = byte1;
    while ( (UCB0IFG & UCTXIFG0) == 0) {}

    //********** Write Byte #2 ***************************
    UCB0TXBUF = byte2;
    while ( (UCB0IFG & UCTXIFG0) == 0) {}

    //stop signal
    UCB0CTLW0 |= UCTXSTP;
    while ( (UCB0CTLW0 & UCTXSTP) != 0) {}

    PJOUT &= ~BIT0;

    return 0;
}

void lcdTriggerEN() {
    P7OUT |= BIT3;
    __delay_cycles(1000);
    P7OUT &= ~BIT3;
}

void lcd_reset()
{

//    //P1OUT = 0x00
//    P1OUT &= ~BIT0;
//    P7OUT &= ~(BIT2|BIT3);
//    P9OUT &= ~(BIT4|BIT5|BIT6);
//    __delay_cycles(20000);

    //SEND 0x3 3 times to reset tthe LCD

    //P1OUT = (0x03 << LCD_D0_OFFSET) | LCD_EN;
    //0011 0100
    P7OUT &= ~(BIT2|BIT3);
    P1OUT |= BIT0;
    P9OUT |= BIT4;
    P9OUT &= ~(BIT5|BIT6);
    lcdTriggerEN();
    __delay_cycles(10000);

    //P1OUT = (0x03 << LCD_D0_OFFSET) | LCD_EN;
    //0011 0100
    P7OUT &= ~(BIT2|BIT3);
    P1OUT |= BIT0;
    P9OUT |= BIT4;
    P9OUT &= ~(BIT5|BIT6);
    lcdTriggerEN();
    __delay_cycles(1000);

    //P1OUT = (0x03 << LCD_D0_OFFSET) | LCD_EN;
    //0011 0100
    P7OUT &= ~(BIT2|BIT3);
    P1OUT |= BIT0;
    P9OUT |= BIT4;
    P9OUT &= ~(BIT5|BIT6);
    lcdTriggerEN();
    __delay_cycles(1000);

//    //P1OUT = (0x02 << LCD_D0_OFFSET) | LCD_EN;
//    //0010 0100
//    P1OUT &= ~BIT0;
//    P9OUT |= BIT4;
//    lcdTriggerEN();
//    __delay_cycles(1000);


}

void lcd_init ()
{
    //WAIT FOR POWER UP
    __delay_cycles(1000);

    lcd_reset();

    //Switch to 4 bit mode
    //send 0010 or 0x2
    //RS = 0
    P7OUT &= ~BIT2;
    P9OUT &= ~(BIT6|BIT5);
    P9OUT |= BIT4;
    P1OUT &= ~BIT0;
    lcdTriggerEN();

    // 4-bit mode - 2 line - 5x8 font.
    //0x28 - 0010 1000
    //0010
    P7OUT &= ~BIT2;
    P1OUT &= ~BIT0;
    P9OUT |= BIT4;
    P9OUT &= ~(BIT5|BIT6);
    lcdTriggerEN();
    //1000
    P7OUT &= ~BIT2;
    P1OUT &= ~BIT0;
    P9OUT |= BIT6;
    P9OUT &= ~(BIT5|BIT4);
    lcdTriggerEN();

    //Display off
    //0x08 - 0000 1000
    //0000
    P7OUT &= ~BIT2;
    P1OUT &= ~BIT0;
    P9OUT &= ~(BIT4|BIT5|BIT6);
    lcdTriggerEN();
    //1000
    P7OUT &= ~BIT2;
    P1OUT &= ~BIT0;
    P9OUT |= BIT6;
    P9OUT &= ~(BIT5|BIT4);
    lcdTriggerEN();

    //Display Clear
    //0x01 - 0000 0001
    //0000
    P7OUT &= ~BIT2;
    P1OUT &= ~BIT0;
    P9OUT &= ~(BIT4|BIT5|BIT6);
    lcdTriggerEN();
    //0001
    P7OUT &= ~BIT2;
    P1OUT |= BIT0;
    P9OUT &= ~(BIT4|BIT5|BIT6);
    lcdTriggerEN();

    //Set Entry Mode
    //0x06 - 0000 0110
    //0000
    P7OUT &= ~BIT2;
    P1OUT &= ~BIT0;
    P9OUT &= ~(BIT4|BIT5|BIT6);
    lcdTriggerEN();
    //0110
    P7OUT &= ~BIT2;
    P1OUT &= ~BIT0;
    P9OUT &= ~BIT6;
    P9OUT |= (BIT5|BIT4);

    //Display ON, no blink
    //0x0C - 0000 1100
    //0000
    P7OUT &= ~BIT2;
    P1OUT &= ~BIT0;
    P9OUT &= ~(BIT4|BIT5|BIT6);
    lcdTriggerEN();
    //1100
    P7OUT &= ~BIT2;
    P1OUT &= ~BIT0;
    P9OUT &= ~(BIT4);
    P9OUT |= (BIT5|BIT6);
    lcdTriggerEN();

}

void clearScreen()
{
    int i = 0;

    //Set address to position 0
    //write command 0x80 - 1000 0000
    //1000
    P7OUT &= ~BIT2;
    P1OUT &= ~BIT0;
    P9OUT |= BIT6;
    P9OUT &= ~(BIT5|BIT4);
    lcdTriggerEN();
    //0000
    P7OUT &= ~BIT2;
    P1OUT &= ~BIT0;
    P9OUT &= ~(BIT4|BIT5|BIT6);
    lcdTriggerEN();

    //Write Spaces over first line
    while (i < 16)
    {
        //write ' ' - - 0010 0000
        //0010
        P7OUT |= BIT2;
        P1OUT &= ~BIT0;
        P9OUT |= BIT4;
        P9OUT &= ~(BIT5|BIT6);
        lcdTriggerEN();
        //0000
        P7OUT |= BIT2;
        P1OUT &= ~BIT0;
        P9OUT &= ~(BIT4|BIT5|BIT6);
        lcdTriggerEN();

        i++;
    }

    //write command 0x80 - 1000 0000
    //1000
    P7OUT &= ~BIT2;
    P1OUT &= ~BIT0;
    P9OUT |= BIT6;
    P9OUT &= ~(BIT5|BIT4);
    lcdTriggerEN();
    //0000
    P7OUT &= ~BIT2;
    P1OUT &= ~BIT0;
    P9OUT &= ~(BIT4|BIT5|BIT6);
    lcdTriggerEN();

}

void displayIDLE()
{

   //Write out Ready on the LCD

    //Write R - 0x52 - 0101 0010
    //0101
    P7OUT |= BIT2;
    P1OUT |= BIT0;
    P9OUT |= BIT5;
    P9OUT &= ~(BIT4|BIT6);
    lcdTriggerEN();
    //0010
    P7OUT |= BIT2;
    P1OUT &= ~BIT0;
    P9OUT &= ~(BIT5|BIT6);
    P9OUT |= BIT4;
    lcdTriggerEN();

    //Write e - 0x65 - 0110 0101
    //0110
    P7OUT |= BIT2;
    P1OUT &= ~BIT0;
    P9OUT |= (BIT5|BIT4);
    P9OUT &= ~(BIT6);
    lcdTriggerEN();
    //0101
    P7OUT |= BIT2;
    P1OUT |= BIT0;
    P9OUT |= BIT5;
    P9OUT &= ~(BIT4|BIT6);
    lcdTriggerEN();

    //Write a - 0110 0001
    //0110
    P7OUT |= BIT2;
    P1OUT &= ~BIT0;
    P9OUT |= (BIT5|BIT4);
    P9OUT &= ~(BIT6);
    lcdTriggerEN();
    //0001
    P7OUT |= BIT2;
    P1OUT |= BIT0;
    P9OUT &= ~(BIT6|BIT5|BIT4);
    lcdTriggerEN();

    //write d - - 0110 0100
    //0110
    P7OUT |= BIT2;
    P1OUT &= ~BIT0;
    P9OUT |= (BIT5|BIT4);
    P9OUT &= ~(BIT6);
    lcdTriggerEN();
    //0100
    P7OUT |= BIT2;
    P1OUT &= ~BIT0;
    P9OUT |= (BIT5);
    P9OUT &= ~(BIT6|BIT4);
    lcdTriggerEN();

    //write y - 0111 1001
    //0111
    P7OUT |= BIT2;
    P1OUT |= BIT0;
    P9OUT |= (BIT5|BIT4);
    P9OUT &= ~(BIT6);
    lcdTriggerEN();
    //1001
    P7OUT |= BIT2;
    P1OUT |= BIT0;
    P9OUT |= (BIT6);
    P9OUT &= ~(BIT5|BIT4);
    lcdTriggerEN();

}

void displayRecording()
{

    //Write out "Recording..." on the LCD

    //Write R - 0x52 - 0101 0010
    //0101
    P7OUT |= BIT2;
    P1OUT |= BIT0;
    P9OUT |= BIT5;
    P9OUT &= ~(BIT4|BIT6);
    lcdTriggerEN();
    //0010
    P7OUT |= BIT2;
    P1OUT &= ~BIT0;
    P9OUT &= ~(BIT5|BIT6);
    P9OUT |= BIT4;
    lcdTriggerEN();

    //Write e - 0x65 - 0110 0101
    //0110
    P7OUT |= BIT2;
    P1OUT &= ~BIT0;
    P9OUT |= (BIT5|BIT4);
    P9OUT &= ~(BIT6);
    lcdTriggerEN();
    //0101
    P7OUT |= BIT2;
    P1OUT |= BIT0;
    P9OUT |= BIT5;
    P9OUT &= ~(BIT4|BIT6);
    lcdTriggerEN();

    //Write c - 0x63 - 0110 0011
    //0110
    P7OUT |= BIT2;
    P1OUT &= ~BIT0;
    P9OUT |= (BIT5|BIT4);
    P9OUT &= ~(BIT6);
    lcdTriggerEN();
    //0011
    P7OUT |= BIT2;
    P1OUT |= BIT0;
    P9OUT |= BIT4;
    P9OUT &= ~(BIT5|BIT6);
    lcdTriggerEN();

    //Write o - - 0110 1111
    //0110
    P7OUT |= BIT2;
    P1OUT &= ~BIT0;
    P9OUT |= (BIT5|BIT4);
    P9OUT &= ~(BIT6);
    lcdTriggerEN();
    //1111
    P7OUT |= BIT2;
    P1OUT |= BIT0;
    P9OUT |= (BIT6|BIT5|BIT4);
    lcdTriggerEN();


    //Write r - - 0111 0010
    //0111
    P7OUT |= BIT2;
    P1OUT |= BIT0;
    P9OUT |= (BIT5|BIT4);
    P9OUT &= ~(BIT6);
    lcdTriggerEN();
    //0010
    P7OUT |= BIT2;
    P1OUT &= ~BIT0;
    P9OUT |= (BIT4);
    P9OUT &= ~(BIT6|BIT5);
    lcdTriggerEN();

    //write d - - 0110 0100
    //0110
    P7OUT |= BIT2;
    P1OUT &= ~BIT0;
    P9OUT |= (BIT5|BIT4);
    P9OUT &= ~(BIT6);
    lcdTriggerEN();
    //0100
    P7OUT |= BIT2;
    P1OUT &= ~BIT0;
    P9OUT |= (BIT5);
    P9OUT &= ~(BIT6|BIT4);
    lcdTriggerEN();

    //write i - - 0110 1001
    //0110
    P7OUT |= BIT2;
    P1OUT &= ~BIT0;
    P9OUT |= (BIT5|BIT4);
    P9OUT &= ~(BIT6);
    lcdTriggerEN();
    //1001
    P7OUT |= BIT2;
    P1OUT |= BIT0;
    P9OUT &= ~(BIT5|BIT4);
    P9OUT |= (BIT6);
    lcdTriggerEN();

    //write n - - 0110 1110
    //0110
    P7OUT |= BIT2;
    P1OUT &= ~BIT0;
    P9OUT |= (BIT5|BIT4);
    P9OUT &= ~(BIT6);
    lcdTriggerEN();
    //1110
    P7OUT |= BIT2;
    P1OUT &= ~BIT0;
    P9OUT |= (BIT6|BIT5|BIT4);
    lcdTriggerEN();

    //write g - - 0110 0111
    //0110
    P7OUT |= BIT2;
    P1OUT &= ~BIT0;
    P9OUT |= (BIT5|BIT4);
    P9OUT &= ~(BIT6);
    lcdTriggerEN();
    //0111
    P7OUT |= BIT2;
    P1OUT |= BIT0;
    P9OUT |= (BIT5|BIT4);
    P9OUT &= ~(BIT6);
    lcdTriggerEN();

    //write . - 0x2E - 0010 1110
    //0010
    P7OUT |= BIT2;
    P1OUT &= ~BIT0;
    P9OUT |= (BIT4);
    P9OUT &= ~(BIT5|BIT6);
    lcdTriggerEN();
    //1110
    P7OUT |= BIT2;
    P1OUT &= ~BIT0;
    P9OUT |= (BIT6|BIT5|BIT4);
    lcdTriggerEN();

    //write . - 0x2E - 0010 1110
    //0010
    P7OUT |= BIT2;
    P1OUT &= ~BIT0;
    P9OUT |= (BIT4);
    P9OUT &= ~(BIT5|BIT6);
    lcdTriggerEN();
    //1110
    P7OUT |= BIT2;
    P1OUT &= ~BIT0;
    P9OUT |= (BIT6|BIT5|BIT4);
    lcdTriggerEN();

    //write . - 0x2E - 0010 1110
    //0010
    P7OUT |= BIT2;
    P1OUT &= ~BIT0;
    P9OUT |= (BIT4);
    P9OUT &= ~(BIT5|BIT6);
    lcdTriggerEN();
    //1110
    P7OUT |= BIT2;
    P1OUT &= ~BIT0;
    P9OUT |= (BIT6|BIT5|BIT4);
    lcdTriggerEN();

}

int main(void)
{
    WDTCTL = WDTPW | WDTHOLD;	// stop watchdog timer
    PM5CTL0 &= ~LOCKLPM5;       // Disable the GPIO power-on default high-impedance mode

    volatile unsigned int ls = BIT1;
    volatile unsigned int i;            // volatile to prevent optimization
    unsigned int data;
	
	//LCD PINS
    P7DIR |= (BIT2 | BIT3);
    P1DIR |= BIT0;
    P9DIR |= (BIT4 | BIT5 | BIT6);


    P7DIR |= (BIT1|BIT0);     //0x0004                     // Set P1.0 to output direction
    P7OUT &= ~(BIT0);
    P7OUT |= BIT1;  //power indicator

    PJDIR |= (BIT0|BIT1|BIT2|BIT3);
    PJOUT &= ~(BIT0|BIT1|BIT2|BIT3);

    //IR indicator
    P3DIR |= BIT3;
    P3OUT &= ~BIT3;

    // P4.5 - Start Button
    P4DIR &= ~BIT5;
    P4REN |= BIT5;
    P4OUT |= BIT5;
    P4IE |= BIT5;
    P4IES |= BIT5;
    P4IFG &= ~BIT5;

    // P3.0 - Stop Button
    P3DIR &= ~BIT0;
    P3REN |= BIT0;
    P3OUT |= BIT0;
    P3IE |= BIT0;
    P3IES |= BIT0;
    P3IFG &= ~BIT0;

//    //P4.4 - IR Data Pin
//    P4DIR &= ~BIT4;
//    P4REN |= BIT4;
//    P4OUT |= BIT4;
//    P4IE |= BIT4;
//    P4IES |= BIT4;
//    P4IFG &= ~BIT4;

    //TESTing P1.1 for IR data
    P1DIR &= ~BIT1;
    P1REN |= BIT1;
    P1OUT |= BIT1;
    P1IE |= BIT1;
    P1IES |= BIT1;
    P1IFG &= ~BIT1;

    // Initialize LCD
    lcd_init();

    //unstick the lines if they are
    clear_i2c();

    //Initialize I2C
    init_I2C();

    displayIDLE();

    __enable_interrupt();

    for(;;) {

        if((P1IN & BIT1) != ls )
        {
            P3OUT ^= BIT3;
            //Send I2C code to arduino
            i2c_write_word(0x8, 0x03);

            //sw delay for debouncing
           for(i=0;i<65000;i++);

           i2c_read_word(0x8, &data);
           if(data == 0x02)
           {
               //turn on LED for a second
               PJOUT |= BIT1;
               //1 second delay
               for(i=0; i< 65000; i++);
               //turn off LED
               PJOUT &= ~BIT1;
           }
        }

        ls = (P1IN & BIT1);


   }
}


#pragma vector = PORT4_VECTOR
__interrupt void Port4_ISR() {

    volatile unsigned int i;            // volatile to prevent optimization
    unsigned static const int code = 0x01;
    unsigned int data = 0;


    if((P4IFG & BIT5) != 0)
        {
            //Turn on Recording LED
            P7OUT |= BIT0;
            //Clear the LCD display
            clearScreen();
            //Display Recording... on LCD
            displayRecording();

            //Send I2C code to arduino
            i2c_write_word(0x8, code);

            //sw delay for debouncing
           for(i=0;i<65000;i++);

          //Wait for response
          i2c_read_word(0x8, &data);
          if(data == 0x02)
                    {
                        //turn on LED for a second
                        PJOUT |= BIT1;
                        //1 second delay
                        for(i=0; i< 65000; i++);
                        //turn off LED
                        PJOUT &= ~BIT1;
                    }

            P4IFG &= ~BIT5;
        }

}

#pragma vector = PORT3_VECTOR
__interrupt void Port3_ISR() {

    volatile unsigned int i;            // volatile to prevent optimization
    unsigned static const int code = 0x00;
    unsigned int data1 = 0;

    if((P3IFG & BIT0) != 0)
        {
            P7OUT &= ~BIT0;
            clearScreen();
            displayIDLE();

            //Send I2C code to arduino
            i2c_write_word(0x8, code);

            //sw delay for debouncing
            for(i=0;i<65000;i++);

            //Wait for response
            i2c_read_word(0x8, &data1);
            if(data1 == 0x02)
                      {
                          //turn on LED for a second
                          PJOUT |= BIT1;
                          //1 second delay
                          for(i=0; i< 65000; i++);
                          //turn off LED
                          PJOUT &= ~BIT1;
                      }



        }

    P3IFG &= ~BIT0;

}

//#pragma vector = PORT1_VECTOR
//__interrupt void Port1_ISR(){
//
//    volatile unsigned int i;            // volatile to prevent optimization
//
//    PJOUT |= BIT0;
//
//   if( P1IFG & BIT1 )
//   {
//       P3OUT |= BIT3;                      // Toggle P1.0 using exclusive-OR
//
//       i = 1000;                          // SW Delay
//       do i--;
//       while(i != 0);
//
//       P1IFG &= ~BIT1;
//       P3OUT &= ~BIT3;
//   }
//
//   PJOUT &= ~BIT0;
//
//}

