﻿using System;
using System.Collections;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using Newtonsoft.Json;
using UnityEngine;

public class update_anim : MonoBehaviour
{
    private Animator anim; // refers to animator controller
    private Hashtable ht; // animation hashmap
    private String[] sentence; // holds tokens for animations
    public static bool startTranslate = false; // used in update_canvas method 
    private Coroutine coroutine; // coroutine handler
    private string[] devices; // holds microphone device names
    private int minFreq; // holds sampling rate for given microphone

    private void Start()
    {
        update_canvas.tmp.SetText(""); // Clear text window
        anim = gameObject.GetComponent<Animator>(); // Get Animator Controller object
        SetHashtable(); // Initialize hash table using vocab.txt file
        devices = Microphone.devices; // Find connected microphones
        Microphone.GetDeviceCaps(devices[0], out minFreq, out _); // Read the minimum sampling rate of that microphone

        // Force program to ask for microphone permissions at start
        Microphone.Start(devices[0], false, 1, minFreq);
        Microphone.End(devices[0]);
    }

    private void Update()
    {
        // PCB "Start" button
        if (Input.GetKeyDown(KeyCode.S))
        {
            update_camera.audioSource.clip = Microphone.Start(devices[0], false, 30, minFreq);
            // Wait until microphone starts recording before prompting user to speak
            while (!Microphone.IsRecording(devices[0])) { } 
            update_canvas.tmp.SetText("Recording...\n");
        }

        // PCB "Stop" button
        if (Input.GetKeyDown(KeyCode.T))
        {
            if (!Microphone.IsRecording(devices[0]))
            {
                micRecording = false; // Bool to enable GUI help message
            }
            else
            {
                Microphone.End(devices[0]);
                update_camera.audioSource.clip = SavWav.TrimSilence(update_camera.audioSource.clip, 0.1f);
                SavWav.Save("watson.wav", update_camera.audioSource.clip); // Write watson.wav audio file
                StartCoroutine(WaitForAudio()); // Time stopping mic with THINKING animation
            }
        }

        // Voice activated? --> Say "End game", pass through watson?
        if (Input.GetKeyDown(KeyCode.Escape))
        {            
            if (!anim.GetCurrentAnimatorStateInfo(0).IsName("IDLE"))
            {
                animInProgress = true; // Bool to enable GUI help message
            }
            else
            {
                string words = "HELLO"; // "goodbye"; similar gesture
                sentence = words.Split(' '); // pass to string array for animating
                StartCoroutine(AnimHandler()); // Start animations
                StartCoroutine(KillGame()); // End game
            }
        }
    }

    private bool animInProgress = false;
    private bool micRecording = true;
    private void OnGUI()
    {
        if (animInProgress)
        {
            GUI.Box(new Rect(10, 10, 250, 20), "Animation in Progress...");
        }
        if (!micRecording)
        {
            GUI.Box(new Rect(10, 30, 250, 20), "Microphone is not recording...");
        }
    }

    private IEnumerator WaitForAudio()
    {
        anim.SetTrigger("THINKING");
        yield return new WaitForSeconds(0.5f);
        Translate(); // Send wav file to Watson API, then OpenNMT program
        update_camera.audioSource.Play();
    }

    // Coroutine to end the game
    private IEnumerator KillGame()
    {
        update_canvas.tmp.SetText("Goodbye! Shutting down...");
        yield return new WaitForSeconds(2f);
        Application.Quit();
    }

    // Coroutine to handle timing of animations 
    private IEnumerator AnimHandler()
    {
        yield return coroutine;
        foreach (string str in sentence)
        {
            coroutine = StartCoroutine(Waitanim(str));
            yield return coroutine;
        }
        update_canvas.tmp.SetText("");
        coroutine = StartCoroutine(Waitanim("IDLE"));
        yield return coroutine;
        animInProgress = false;
    }

    // Coroutine to handle string parsing and animation mapping
    // If a string is in the hash table, the corresponding animation is played.
    // Otherwise, the string is split into characters and fingerspelled. 
    private bool isInWaitanim = false; // Used as debouncing bool
    private IEnumerator Waitanim(string clip)
	{
		if (isInWaitanim)
            yield break;
		isInWaitanim = true;
        if (ht.Contains(clip))
        { // Case: gesture in hash map
            anim.SetTrigger(ht[clip].GetHashCode());
            float time = GetTimestamp(clip);
            UnityEngine.Debug.Log("Playing " + clip + " for " + time + " seconds...");
            yield return new WaitForSecondsRealtime(time);
        }
        else
        { // Case: fingerspelling
            char[] word = clip.ToCharArray();
            foreach(char letter in word)
            {
                anim.SetTrigger(letter.ToString());
                float time = GetTimestamp(letter.ToString());
                UnityEngine.Debug.Log("Playing " + letter.ToString() + " for " + time + " seconds...");
                yield return new WaitForSecondsRealtime(time);
            }
        }
        isInWaitanim = false;
	}

    // Initializes hashtable
    private readonly int hashsize = 92;
    private void SetHashtable()
    {
        // Create Hashtable from vocab.txt file
        string[] AllLines = new string[hashsize];
        ht = new Hashtable();
        string fileName = "ASL\\vocab.txt";
        using (StreamReader sr = File.OpenText(fileName))
        {
            int x = 0;
            int stateName;
            while (!sr.EndOfStream)
            {
                AllLines[x] = sr.ReadLine();
                x += 1;
            }
            foreach (string str in AllLines)
            {
                if (!ht.ContainsKey(str))
                {
                    stateName = Animator.StringToHash(str);
                    ht.Add(str, stateName);
                }
            }
        }
    }

    // Returns wait time for each animation
    private float GetTimestamp(string input)
    {
        AnimationClip[] clips = anim.runtimeAnimatorController.animationClips;
        foreach(AnimationClip clip in clips)
            if (clip.name.Equals(input))
                return clip.length;
        return 0f;
    }

    // Take the WAV file to send through Watson STT API, get JSON file
    // Take JSON file, parse to TXT file to send through OpenNMT, get another TXT file
    // Map translated TXT file to the sentence string array
    private void Translate()
    {
        /* Watson Speech-to-Text Commands
        // Transcribe speech from watson.wav. Generate text in translate.txt.
        */
        CallProcess("E:\\Anaconda\\Library\\bin\\curl.exe", "-X POST -u \"apikey:0vGNMplrnKJ4wyqVfto2YwkZF-8JTePUcmZSUH9YuT0g\" --header \"Content-Type: audio/wav\" --data-binary @ASL\\watson.wav \"https://api.us-south.speech-to-text.watson.cloud.ibm.com/instances/e0a4a3e1-40a8-45ed-be4d-1a8be797a017/v1/recognize\"");
        string jsonfile = File.ReadAllText("ASL\\watson.json"); // pass watson.json file to string
        output json = JsonConvert.DeserializeObject<output>(jsonfile); // Deserialize string to type <output>
        string words = json.results[0].alternatives[0].transcript; // Grab transcript string object from output
        words = words.ToLower(); // shift lowercase to prevent OpenNMT translation errors
        sentence = words.Split(' '); // pass to string array for word checking
        words = ""; // clear string to prepare for creating translate.txt file

        // Capitalize certain words in the string, e.g. "I"
        //      to prevent OpenNMT translation errors
        for (int i = 0; i < sentence.Length; i++) 
        {
            // case: "i" mapped to "I"
            if (sentence[i].Equals("i"))
                sentence[i] = "I";
            // case: remove %HESITATION markers from Watson STT
            if (!sentence[i].Equals("%hesitation")) 
                words += sentence[i] + " ";
        }
        File.WriteAllText("ASL\\translate.txt", words); // Write transcript string to translate.txt

        /* OpenNMT model commands
        // Translate sentence in translate.txt. Generate translation in pred.txt.
        */
        CallProcess("E:\\Anaconda\\Scripts\\onmt_translate.exe", "-model ASL\\nmt_model.pt -src ASL\\translate.txt -output ASL\\pred.txt -replace_unk -verbose");
        String sen = File.ReadAllText("ASL\\pred.txt").ToUpper(); // pass pred.txt file to string; force uppercase to prevent animation mismapping
        startTranslate = true; // tell update_canvas to start displaying text
        sentence = sen.Split(' '); // pass to final string array for animation mapping
        sentence[sentence.Length - 1] = sentence[sentence.Length - 1].Remove(sentence[sentence.Length - 1].Length - 1); // remove "\n" at end of last word in json file

        // ASL/pred.txt deleted in update_canvas to prevent premature canvas clearing
        // ASL/translate.txt deleted in update_canvas to prevent premature canvas clearing
        File.Delete("ASL\\watson.wav"); // Remove audio file to prepare for next audio acquisition
        File.Delete("ASL\\watson.json"); // Remove Watson STT output to prepare for next API call

        StartCoroutine(AnimHandler()); // Start animation mapping using sentence[]
    }

    // Use to run any kind of terminal commands.
    private int count = 1;
    private void CallProcess(string path, string argument)
    {
        Stopwatch stopwatch = new Stopwatch();
        ProcessStartInfo startInfo = new ProcessStartInfo()
        {
            FileName = path,
            UseShellExecute = false,
            RedirectStandardError = true,
            RedirectStandardInput = true,
            RedirectStandardOutput = true,
            CreateNoWindow = true,
            Arguments = argument
        };
        Process myProcess = new Process
        {
            StartInfo = startInfo
        };
        stopwatch.Start();
        myProcess.Start();
        string output = myProcess.StandardOutput.ReadToEnd();
        if (path.Contains("curl") && argument.Contains("apikey")) // Write to watson.json after using Watson STT
            File.WriteAllText("ASL\\watson.json", output);
        myProcess.WaitForExit();
        stopwatch.Stop();
        TimeSpan time = stopwatch.Elapsed;
        string elapsedTime = String.Format("{0:00}:{1:00}:{2:00}.{3:00}", time.Hours, time.Minutes, time.Seconds, time.Milliseconds / 10);
        File.WriteAllText("timestamp" + count.ToString() + ".txt", path + " " + argument + "\n" + elapsedTime);
        count++;
    }
}

// JSON declassifier parameters
class output
{
    public List<results> results { get; set; }
    public int result_index { get; set; }
}
class results
{
    public List<alternatives> alternatives { get; set; }
    public bool final { get; set; }
}
class alternatives
{
    public float confidence { get; set; }
    public string transcript { get; set; }
}