/* USER CODE BEGIN Header */
/**
 ******************************************************************************
 * @file           : main.c
 * @brief          : Main program body
 ******************************************************************************
 * @attention
 *
 * <h2><center>&copy; Copyright (c) 2020 STMicroelectronics.
 * All rights reserved.</center></h2>
 *
 * This software component is licensed by ST under BSD 3-Clause license,
 * the "License"; You may not use this file except in compliance with the
 * License. You may obtain a copy of the License at:
 *                        opensource.org/licenses/BSD-3-Clause
 *
 ******************************************************************************
 */
/* USER CODE END Header */

/* Includes ------------------------------------------------------------------*/
#include "main.h"

/* Private includes ----------------------------------------------------------*/
/* USER CODE BEGIN Includes */
#include "tacocat_constants.h"
/* USER CODE END Includes */

/* Private typedef -----------------------------------------------------------*/
/* USER CODE BEGIN PTD */
typedef struct NeuronLayer {
	int inputsPerNeuron;
	int neuronCount;
	uint8_t **weights;
} NeuronLayer;

typedef struct NeuralNetwork {
	int layerCount;
	NeuronLayer **layers;
	uint8_t *weights;
	int weightCount;
	int neuronCount;
} NeuralNetwork;
/* USER CODE END PTD */

/* Private define ------------------------------------------------------------*/
/* USER CODE BEGIN PD */
// Neural network dimensions
#define NEURON_COUNT 6
#define WEIGHT_COUNT 309
#define INPUT_LAYER_SIZE 100

// I2C macros

#define SLAVE_ADDRESS 0x04

#define INITIALIZE_CMD 0x01
#define SET_WEIGHTS_CMD 0x02
#define SET_INPUTS_CMD 0x03
#define READ_OUTPUTS_CMD 0x05
#define READ_DB_SUMS_CMD 0x06

//PB6     ------> I2C1_SCL
#define I2C1_SCL_PORT GPIOB
#define I2C1_SCL_PIN GPIO_PIN_6
//PB7     ------> I2C1_SDA
#define I2C1_SDA_PORT GPIOB
#define I2C1_SDA_PIN GPIO_PIN_7

// Digital pot macros

#define SPI_CLOCK_RATE 4000000  // MCP40210 max clock rate for daisy-chain configuration is 5.8 MHz
#define WRITE_COMMAND 0x10
#define P0_WRITE_COMMAND WRITE_COMMAND | 0x01
#define P1_WRITE_COMMAND WRITE_COMMAND | 0x02

// Pin selection macros

#define CHIP_SELECT_PIN 9
/* USER CODE END PD */

/* Private macro -------------------------------------------------------------*/
/* USER CODE BEGIN PM */

/* USER CODE END PM */

/* Private variables ---------------------------------------------------------*/
ADC_HandleTypeDef hadc1;
ADC_HandleTypeDef hadc3;

I2C_HandleTypeDef hi2c1;

TIM_HandleTypeDef htim7;

/* USER CODE BEGIN PV */

// I2C timeout in milliseconds
const int _i2c_timeout = 100;

// Output read pins
const uint8_t _outputReadPins[NEURON_COUNT] = {33,34,35,36,37,38};

uint8_t inputs[INPUT_LAYER_SIZE];
uint16_t outputs[NEURON_COUNT] = {1,2,3,4,5,6};
uint8_t weights[WEIGHT_COUNT];

uint16_t daughterboard_sums[3] = {0,0,0};

// Set up MLP layers.
uint8_t *layer_0_weights_by_neuron[3];
NeuronLayer layer0 = { .inputsPerNeuron = 100, .neuronCount = 3, .weights = layer_0_weights_by_neuron };

uint8_t *layer_1_weights_by_neuron[3];
NeuronLayer layer1 = { .inputsPerNeuron = 3, .neuronCount = 3, .weights = layer_1_weights_by_neuron };

NeuronLayer *layers[] = { &layer0, &layer1 };
NeuralNetwork mlp = { .layerCount = 2, .layers = layers, .weights = weights, .weightCount = WEIGHT_COUNT, .neuronCount = NEURON_COUNT };
NeuralNetwork *MLP = &mlp;

const uint8_t _P0WriteCommand = P0_WRITE_COMMAND;
const uint8_t _P1WriteCommand = P1_WRITE_COMMAND;
const uint8_t _default_weight = 0;

uint8_t rxBuffer[110];
uint32_t outputReadTime;

/* USER CODE END PV */

/* Private function prototypes -----------------------------------------------*/
void SystemClock_Config(void);
static void MX_GPIO_Init(void);
static void MX_I2C1_Init(void);
static void MX_ADC1_Init(void);
static void MX_ADC3_Init(void);
static void MX_TIM7_Init(void);
/* USER CODE BEGIN PFP */

void initializeNeuralNetwork();
void readOutputs();
void readDaughterBoardSums();
void receiveCommand(int byteCount);
void receiveWeights(int byteCount);
void setInputPins(uint8_t *data, int byteCount);
void sendOutputData();
void sendDaughterboardSumData();
void sendLayer_0_Weights();
void sendLayer_1_Weights();
void sendWeights(
		uint8_t* weights,
		int length,
		GPIO_TypeDef *csPort,
		int32_t csPinNumber,
		SPI_TypeDef *spi
);
void setDigitalPots(uint8_t data[], int length);
void softwareSPI_TransmitWord(
		uint16_t tx_word,
		GPIO_TypeDef *clkPort,
		int32_t clkPinNumber,
		GPIO_TypeDef *mosiPort,
		int32_t mosiPinNumber
);
void softwareSPI_SendWeights(
		uint8_t* data,
		int length,
		GPIO_TypeDef *clkPort,
		int32_t clkPinNumber,
		GPIO_TypeDef *csPort,
		int32_t csPinNumber,
		GPIO_TypeDef *mosiPort,
		int32_t mosiPinNumber
);
void I2C1_ClearBusyFlagErratum(I2C_HandleTypeDef *instance);
/* USER CODE END PFP */

/* Private user code ---------------------------------------------------------*/
/* USER CODE BEGIN 0 */
void initializeNeuralNetwork()
{
	int weightOffset = 0;

	for (int layerIdx = 0; layerIdx < MLP->layerCount; layerIdx++)
	{
		NeuronLayer *layer = MLP->layers[layerIdx];

		for (int neuronIdx = 0; neuronIdx < layer->neuronCount; neuronIdx++)
		{
			layer->weights[neuronIdx] = weights + weightOffset;
			weightOffset += layer->inputsPerNeuron;
		}
	}

	for (int i = 0; i < MLP->weightCount; i++)
		weights[i] = _default_weight;
}

void readOutputs()
{
	HAL_StatusTypeDef result;
	for (int i = 0; i < 6; i++)
	{
		result = HAL_ADC_ConfigChannel(OutputADC_Handles[i], OutputADC_ChannelConfigs[i]);

		result = HAL_ADC_Start(OutputADC_Handles[i]);

		result = HAL_ADC_PollForConversion(OutputADC_Handles[i], 100);

		outputs[i] = HAL_ADC_GetValue(OutputADC_Handles[i]);

		result = HAL_ADC_Stop(OutputADC_Handles[i]);

	}
}

void readDaughterBoardSums()
{
	HAL_StatusTypeDef result;
	for (int i = 0; i < 3; i++)
	{
		result = HAL_ADC_ConfigChannel(DaughterboardSumADC_Handles[i], DaughterboardSumADC_ChannelConfigs[i]);

		result = HAL_ADC_Start(DaughterboardSumADC_Handles[i]);

		result = HAL_ADC_PollForConversion(DaughterboardSumADC_Handles[i], 100);

		daughterboard_sums[i] = HAL_ADC_GetValue(DaughterboardSumADC_Handles[i]);

		result = HAL_ADC_Stop(DaughterboardSumADC_Handles[i]);
	}
}

void setInputPins(uint8_t *dataBuffer, int byteCount)
{
	// Bitmask is set to 0x80 for reading input value from MSB.
	const uint8_t bitMask = 0x80;
	int inputIdx = 0;

	// Unpack input values from bit fields.
	for (int i = 0; i <= INPUT_LAYER_SIZE/8; i++)
	{
		uint8_t data = dataBuffer[i];

		for (int j = 0; j < 8 && inputIdx < INPUT_LAYER_SIZE; j++)
		{

			if ((data & bitMask) != 0)
			{
				// Set the pin to output mode with active state.
				Pixel_Drive_Ports[inputIdx]->MODER &= ~Pixel_Drive_Pin_11_Mode_Masks[inputIdx];
				Pixel_Drive_Ports[inputIdx]->MODER |=  Pixel_Drive_Pin_01_Mode_Masks[inputIdx];
			}
			else
			{
				// Set the pin mode to 'Analog Input' so that it can be pulled to 0V by its
				//  external pull-down resistor.
				Pixel_Drive_Ports[inputIdx]->MODER |=  Pixel_Drive_Pin_11_Mode_Masks[inputIdx];
			}

			data <<= 1;
			inputIdx += 1;
		}
	}
}

void sendOutputData()
{
//	readOutputs();

	uint8_t *bytes = (uint8_t*)outputs;

	HAL_I2C_Slave_Transmit(&hi2c1, bytes, 12, _i2c_timeout);
}

void sendDaughterboardSumData()
{
	readDaughterBoardSums();

	uint8_t *bytes = (uint8_t*)daughterboard_sums;

	HAL_I2C_Slave_Transmit(&hi2c1, bytes, 6, _i2c_timeout);
}

void sendWeights(
		uint8_t* weights,
		int length,
		GPIO_TypeDef *csPort,
		int32_t csPinNumber,
		SPI_TypeDef *spi
)
{
	uint32_t data_buffer;

	const int pots_per_chip = 4;
	const int reg_width = 11;
	const int addr_width = 3;
	const int data_width = 8;
	const int buffer_width = 32;
	const int initial_left_shift = buffer_width - addr_width - (8 - (length*reg_width) % 8);

	int csMask = 1 << csPinNumber;
	int boundary;

	uint8_t dummyRxBuffer;

	for (int addr = 0; addr < pots_per_chip; addr++)
	{
		boundary = buffer_width-1 - initial_left_shift;

		data_buffer = (addr << initial_left_shift);
		boundary += data_width;
		data_buffer |= ((uint32_t)weights[addr] << (31-boundary));

		// Pull CS down to activate.
		csPort->ODR &= ~csMask;

		for (int i = addr+pots_per_chip; boundary > 0; )
		{
			uint8_t tx_byte = (data_buffer & 0xff000000) >> 24;
			data_buffer <<= 8;
			boundary -= 8;

			// Transmit one byte over SPI.
			LL_SPI_TransmitData16(spi, (uint32_t)5);//tx_byte);

			if (boundary < 8 && i < length)
			{
				int32_t new_data = (addr << data_width) | weights[i];

				i += pots_per_chip;

				int shift_amt = buffer_width-1 - addr_width - data_width - boundary;

				data_buffer |= new_data << shift_amt;
				boundary += addr_width + data_width;
			}
		}

		csPort->ODR |= csMask;
	}
}

void setDigitalPots(uint8_t data[], int length) // Digital pot code for MCP42010
{
	//  // Write all of the values for the P0 pots.
	//  SPI.beginTransaction(SPISettings(SPI_CLOCK_RATE, MSBFIRST, SPI_MODE0));
	//  digitalWrite(CHIP_SELECT_PIN, LOW);
	//  for (int i = 0; i < length; i += 2)
	//  {
	//    SPI.transfer(_P0WriteCommand);
	//    SPI.transfer(data[i]);
	//  }
	//  digitalWrite(CHIP_SELECT_PIN, HIGH);
	//  SPI.endTransaction();
	//
	//  // Write all of the values for the P1 pots.
	//  SPI.beginTransaction(SPISettings(SPI_CLOCK_RATE, MSBFIRST, SPI_MODE0));
	//  digitalWrite(CHIP_SELECT_PIN, LOW);
	//  for (int j = 1; j < length; j += 2)
	//  {
	//    SPI.transfer(_P1WriteCommand);
	//    SPI.transfer(data[j]);
	//  }
	//  digitalWrite(CHIP_SELECT_PIN, HIGH);
	//  SPI.endTransaction();
	//
	//  delayMicroseconds(10);
}

void softwareSPI_TransmitWord(
		uint16_t tx_word,
		GPIO_TypeDef *clkPort,
		int32_t clkPinNumber,
		GPIO_TypeDef *mosiPort,
		int32_t mosiPinNumber
)
{
	const uint8_t wordSize = 11;
	const uint16_t bitMask =  1 << (wordSize-1);

	const int idleCycles = 1;

	int clkMask = 1 << clkPinNumber;
	int mosiMask = 1 << mosiPinNumber;

	clkPort->ODR &= ~clkMask;

	// Transmit a word using bit-bang/software SPI.
	for (int i = 0; i < wordSize; i++)
	{
		if (((tx_word << i) & bitMask) != 0)
			mosiPort->ODR |= mosiMask;
		else
			mosiPort->ODR &= ~mosiMask;

		for (volatile int i = 0; i < idleCycles; i++);

		clkPort->ODR |= clkMask;

		for (volatile int i = 0; i < idleCycles; i++);

		clkPort->ODR &= ~clkMask;
	}
}

void softwareSPI_SendWeights(
		uint8_t* data,
		int length,
		GPIO_TypeDef *clkPort,
		int32_t clkPinNumber,
		GPIO_TypeDef *csPort,
		int32_t csPinNumber,
		GPIO_TypeDef *mosiPort,
		int32_t mosiPinNumber
)
{
	uint32_t data_buffer;

	const int pots_per_chip = 4;
	const int reg_width = 11;
	const int addr_width = 3;
	const int data_width = 8;

	const int idleCycles = 1;

	int boundary;

	int clkMask = 1 << clkPinNumber;
	int csMask = 1 << csPinNumber;


	// Make sure CS line is pulled up before starting transmission.
	csPort->ODR |= csMask;

	for (int addr = 0; addr < pots_per_chip; addr++)
	{
		// Make sure the clock is pulled down before activating the CS line.
		clkPort->ODR &= ~clkMask;

		for (volatile int i = 0; i < idleCycles; i++);

		// Pull CS down to activate.
		csPort->ODR &= ~csMask;

		for (int i = addr; i < length; i+=pots_per_chip)
		{
			uint16_t tx_word = (addr << data_width) | data[i];
			softwareSPI_TransmitWord(tx_word, clkPort, clkPinNumber, mosiPort, mosiPinNumber);
		}

		csPort->ODR |= csMask;

		for (volatile int i = 0; i < idleCycles; i++);
	}
}

void sendLayer_0_Weights()
{
	softwareSPI_SendWeights(MLP->layers[0]->weights[0], 100, GPIOC, 10, GPIOB, 1, GPIOB, 2);
	softwareSPI_SendWeights(MLP->layers[0]->weights[1], 100, GPIOC, 10, GPIOB, 0, GPIOB, 2);
	softwareSPI_SendWeights(MLP->layers[0]->weights[2], 100, GPIOC, 10, GPIOC, 5, GPIOB, 2);
}

void sendLayer_1_Weights()
{
	uint8_t layer1_weights[12];
	int idx = 0;
	const int chipOrder[] = {0,2,1};

	for (int i = 0; i < 3; i++)
	{
		// Pot and input labels start at "1"

		// Pot 1 is unused
		layer1_weights[idx++] = 127;

		// Pot 2 --> Input 2
		layer1_weights[idx++] = MLP->layers[1]->weights[chipOrder[i]][2 - 1];

		// Pot 3 --> Input 3
		layer1_weights[idx++] = MLP->layers[1]->weights[chipOrder[i]][3 - 1];

		// Pot 4 --> Input 1
		layer1_weights[idx++] = MLP->layers[1]->weights[chipOrder[i]][1 - 1];

	}

	softwareSPI_SendWeights(layer1_weights, 12, GPIOA, 5, GPIOA, 6, GPIOA, 7);
}

/* USER CODE END 0 */

/**
  * @brief  The application entry point.
  * @retval int
  */
int main(void)
{
  /* USER CODE BEGIN 1 */

  /* USER CODE END 1 */

  /* MCU Configuration--------------------------------------------------------*/

  /* Reset of all peripherals, Initializes the Flash interface and the Systick. */
  HAL_Init();

  /* USER CODE BEGIN Init */

  /* USER CODE END Init */

  /* Configure the system clock */
  SystemClock_Config();

  /* USER CODE BEGIN SysInit */

  /* USER CODE END SysInit */

  /* Initialize all configured peripherals */
  MX_GPIO_Init();
  MX_I2C1_Init();
  MX_ADC1_Init();
  MX_ADC3_Init();
  MX_TIM7_Init();
  /* USER CODE BEGIN 2 */
	initializeNeuralNetwork();

	initializeAdcChannels();

	// Set pins to Analog (Hi-Z) mode.
	for (int i = 0; i < 100; i++)
	{
		Pixel_Drive_Ports[i]->OTYPER &= ~(1 << Pixel_Drive_Pin_Numbers[i]);
		Pixel_Drive_Ports[i]->ODR |= (1 << Pixel_Drive_Pin_Numbers[i]);

		Pixel_Drive_Ports[i]->MODER |=  Pixel_Drive_Pin_11_Mode_Masks[i];

		Pixel_Drive_Ports[i]->OSPEEDR |=  Pixel_Drive_Pin_11_Mode_Masks[i];
	}

	__TIM7_CLK_ENABLE();

	//	 // Begin Test Code 0

//	// Set pins to active digital output state.
//	for (int i = 0; i < 40; i++)
//	{
//		Pixel_Drive_Ports[i]->OTYPER &= ~(1 << Pixel_Drive_Pin_Numbers[i]);
//		Pixel_Drive_Ports[i]->ODR |= (1 << Pixel_Drive_Pin_Numbers[i]);
//
//		Pixel_Drive_Ports[i]->MODER &= ~Pixel_Drive_Pin_11_Mode_Masks[i];
//		Pixel_Drive_Ports[i]->MODER |=  Pixel_Drive_Pin_01_Mode_Masks[i];
//	}
//
//	sendLayer_0_Weights();
//	sendLayer_1_Weights();

//	 // End Test Code 0

  /* USER CODE END 2 */

  /* Infinite loop */
  /* USER CODE BEGIN WHILE */
	while (1)
	{
		HAL_StatusTypeDef result;
		if ((result = HAL_I2C_Slave_Receive(&hi2c1, rxBuffer, 3, 500)) == HAL_OK)
		{
			uint8_t command = rxBuffer[0];

			uint16_t length = *((uint16_t *)(rxBuffer+1));

			if (command == SET_WEIGHTS_CMD)
			{
				HAL_StatusTypeDef result;
				result = HAL_I2C_Slave_Receive(&hi2c1, MLP->weights, length, _i2c_timeout);
//				if ((result = HAL_I2C_Slave_Receive(&hi2c1, MLP->weights, length, _i2c_timeout)) != HAL_OK)
//				{
//					continue;
//				}

				sendLayer_0_Weights();
				sendLayer_1_Weights();
			}

			else if (command == SET_INPUTS_CMD)
			{
				if (HAL_I2C_Slave_Receive(&hi2c1, rxBuffer, length, _i2c_timeout) != HAL_OK)
				{
					continue;
				}

				htim7.Instance->CNT = 0;
				HAL_TIM_Base_Start(&htim7);

				setInputPins(rxBuffer, MLP->layers[0]->inputsPerNeuron);

				readOutputs();

				outputReadTime = htim7.Instance->CNT;
				HAL_TIM_Base_Stop(&htim7);
			}

			else if (command == READ_OUTPUTS_CMD)
			{
				sendOutputData();
			}

			else if (command == READ_DB_SUMS_CMD)
			{
				sendDaughterboardSumData();
			}
		}
		else
		{
//			HAL_StatusTypeDef result = HAL_I2C_IsDeviceReady(&hi2c1, 4, 5, 1);
//			while(result == HAL_BUSY)
//			{
//				I2C1_ClearBusyFlagErratum(&hi2c1);
//				result = HAL_I2C_IsDeviceReady(&hi2c1, 4, 5, 1);
//			}
		}



		/*
		//BEGIN Digi-pot sweep

		// Set output layer synapses
		for (int i = 0; i < 1; i++)
		{
//			uint8_t output_weights[12] = {127,127,127,127,127,127,127,127,127,127,127,127,127,127,127,127};
			uint8_t output_weights[12] = {255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255};
			sendLayer_1_weights();
		}

		// Sweep pots up.
		for (int i = 0; i < 100; i++)
		{
			for (int j = 0; j < 256; j+=5)
			{
				MLP->layers[0]->weights[0][i] = j;
				sendLayer_0_Weights();
			}

			readOutputs();
		}

		// Sweep pots down.
		for (int i = 0; i < 100; i++)
		{
			for (int j = 255; j >= 0; j-=5)
			{
				MLP->layers[0]->weights[0][i] = j;
				sendLayer_0_Weights();
			}

			readOutputs();
		}

		//END Digi-pot sweep
		*/
		//	  //BEGIN Output pin sweep
		//
		//	  // Set pins to Analog (Hi-Z) mode.
		//	  for (int i = 0; i < 100; i++)
		//	  {
		//		  Pixel_Drive_Ports[i]->MODER |=  Pixel_Drive_Pin_11_Mode_Masks[i];
		//
		//		  HAL_Delay(10);
		//	  }
		//
		//	  // Set pins to active digital output state.
		//	  for (int i = 0; i < 100; i++)
		//	  {
		//		  Pixel_Drive_Ports[i]->MODER &= ~Pixel_Drive_Pin_11_Mode_Masks[i];
		//		  Pixel_Drive_Ports[i]->MODER |=  Pixel_Drive_Pin_01_Mode_Masks[i];
		//
		//		  HAL_Delay(10);
		//	  }
		//
		//	  //END Output pin sweep

    /* USER CODE END WHILE */

    /* USER CODE BEGIN 3 */
	}
  /* USER CODE END 3 */
}

/**
  * @brief System Clock Configuration
  * @retval None
  */
void SystemClock_Config(void)
{
  RCC_OscInitTypeDef RCC_OscInitStruct = {0};
  RCC_ClkInitTypeDef RCC_ClkInitStruct = {0};
  RCC_PeriphCLKInitTypeDef PeriphClkInitStruct = {0};

  /** Supply configuration update enable 
  */
  HAL_PWREx_ConfigSupply(PWR_LDO_SUPPLY);
  /** Configure the main internal regulator output voltage 
  */
  __HAL_PWR_VOLTAGESCALING_CONFIG(PWR_REGULATOR_VOLTAGE_SCALE0);

  while(!__HAL_PWR_GET_FLAG(PWR_FLAG_VOSRDY)) {}
  /** Macro to configure the PLL clock source 
  */
  __HAL_RCC_PLL_PLLSOURCE_CONFIG(RCC_PLLSOURCE_HSE);
  /** Initializes the CPU, AHB and APB busses clocks 
  */
  RCC_OscInitStruct.OscillatorType = RCC_OSCILLATORTYPE_HSE;
  RCC_OscInitStruct.HSEState = RCC_HSE_ON;
  RCC_OscInitStruct.PLL.PLLState = RCC_PLL_ON;
  RCC_OscInitStruct.PLL.PLLSource = RCC_PLLSOURCE_HSE;
  RCC_OscInitStruct.PLL.PLLM = 1;
  RCC_OscInitStruct.PLL.PLLN = 120;
  RCC_OscInitStruct.PLL.PLLP = 2;
  RCC_OscInitStruct.PLL.PLLQ = 2;
  RCC_OscInitStruct.PLL.PLLR = 4;
  RCC_OscInitStruct.PLL.PLLRGE = RCC_PLL1VCIRANGE_3;
  RCC_OscInitStruct.PLL.PLLVCOSEL = RCC_PLL1VCOWIDE;
  RCC_OscInitStruct.PLL.PLLFRACN = 0;
  if (HAL_RCC_OscConfig(&RCC_OscInitStruct) != HAL_OK)
  {
    Error_Handler();
  }
  /** Initializes the CPU, AHB and APB busses clocks 
  */
  RCC_ClkInitStruct.ClockType = RCC_CLOCKTYPE_HCLK|RCC_CLOCKTYPE_SYSCLK
                              |RCC_CLOCKTYPE_PCLK1|RCC_CLOCKTYPE_PCLK2
                              |RCC_CLOCKTYPE_D3PCLK1|RCC_CLOCKTYPE_D1PCLK1;
  RCC_ClkInitStruct.SYSCLKSource = RCC_SYSCLKSOURCE_PLLCLK;
  RCC_ClkInitStruct.SYSCLKDivider = RCC_SYSCLK_DIV1;
  RCC_ClkInitStruct.AHBCLKDivider = RCC_HCLK_DIV2;
  RCC_ClkInitStruct.APB3CLKDivider = RCC_APB3_DIV2;
  RCC_ClkInitStruct.APB1CLKDivider = RCC_APB1_DIV2;
  RCC_ClkInitStruct.APB2CLKDivider = RCC_APB2_DIV2;
  RCC_ClkInitStruct.APB4CLKDivider = RCC_APB4_DIV2;

  if (HAL_RCC_ClockConfig(&RCC_ClkInitStruct, FLASH_LATENCY_4) != HAL_OK)
  {
    Error_Handler();
  }
  PeriphClkInitStruct.PeriphClockSelection = RCC_PERIPHCLK_ADC|RCC_PERIPHCLK_I2C1;
  PeriphClkInitStruct.PLL3.PLL3M = 1;
  PeriphClkInitStruct.PLL3.PLL3N = 19;
  PeriphClkInitStruct.PLL3.PLL3P = 3;
  PeriphClkInitStruct.PLL3.PLL3Q = 2;
  PeriphClkInitStruct.PLL3.PLL3R = 2;
  PeriphClkInitStruct.PLL3.PLL3RGE = RCC_PLL3VCIRANGE_3;
  PeriphClkInitStruct.PLL3.PLL3VCOSEL = RCC_PLL3VCOMEDIUM;
  PeriphClkInitStruct.PLL3.PLL3FRACN = 0;
  PeriphClkInitStruct.I2c123ClockSelection = RCC_I2C123CLKSOURCE_D2PCLK1;
  PeriphClkInitStruct.AdcClockSelection = RCC_ADCCLKSOURCE_PLL3;
  if (HAL_RCCEx_PeriphCLKConfig(&PeriphClkInitStruct) != HAL_OK)
  {
    Error_Handler();
  }
}

/**
  * @brief ADC1 Initialization Function
  * @param None
  * @retval None
  */
static void MX_ADC1_Init(void)
{

  /* USER CODE BEGIN ADC1_Init 0 */

  /* USER CODE END ADC1_Init 0 */

  ADC_MultiModeTypeDef multimode = {0};
  ADC_ChannelConfTypeDef sConfig = {0};

  /* USER CODE BEGIN ADC1_Init 1 */

  /* USER CODE END ADC1_Init 1 */
  /** Common config 
  */
  hadc1.Instance = ADC1;
  hadc1.Init.ClockPrescaler = ADC_CLOCK_ASYNC_DIV1;
  hadc1.Init.Resolution = ADC_RESOLUTION_16B;
  hadc1.Init.ScanConvMode = ADC_SCAN_DISABLE;
  hadc1.Init.EOCSelection = ADC_EOC_SINGLE_CONV;
  hadc1.Init.LowPowerAutoWait = DISABLE;
  hadc1.Init.ContinuousConvMode = DISABLE;
  hadc1.Init.NbrOfConversion = 1;
  hadc1.Init.DiscontinuousConvMode = ENABLE;
  hadc1.Init.NbrOfDiscConversion = 1;
  hadc1.Init.ExternalTrigConv = ADC_SOFTWARE_START;
  hadc1.Init.ExternalTrigConvEdge = ADC_EXTERNALTRIGCONVEDGE_NONE;
  hadc1.Init.ConversionDataManagement = ADC_CONVERSIONDATA_DR;
  hadc1.Init.Overrun = ADC_OVR_DATA_PRESERVED;
  hadc1.Init.LeftBitShift = ADC_LEFTBITSHIFT_NONE;
  hadc1.Init.OversamplingMode = ENABLE;
  hadc1.Init.Oversampling.Ratio = 8;
  hadc1.Init.Oversampling.RightBitShift = ADC_RIGHTBITSHIFT_3;
  hadc1.Init.Oversampling.TriggeredMode = ADC_TRIGGEREDMODE_SINGLE_TRIGGER;
  hadc1.Init.Oversampling.OversamplingStopReset = ADC_REGOVERSAMPLING_CONTINUED_MODE;
  if (HAL_ADC_Init(&hadc1) != HAL_OK)
  {
    Error_Handler();
  }
  /** Configure the ADC multi-mode 
  */
  multimode.Mode = ADC_MODE_INDEPENDENT;
  if (HAL_ADCEx_MultiModeConfigChannel(&hadc1, &multimode) != HAL_OK)
  {
    Error_Handler();
  }
  /** Configure Regular Channel 
  */
  sConfig.Channel = ADC_CHANNEL_15;
  sConfig.Rank = ADC_REGULAR_RANK_1;
  sConfig.SamplingTime = ADC_SAMPLETIME_8CYCLES_5;
  sConfig.SingleDiff = ADC_SINGLE_ENDED;
  sConfig.OffsetNumber = ADC_OFFSET_NONE;
  sConfig.Offset = 0;
  if (HAL_ADC_ConfigChannel(&hadc1, &sConfig) != HAL_OK)
  {
    Error_Handler();
  }
  /* USER CODE BEGIN ADC1_Init 2 */

  /* USER CODE END ADC1_Init 2 */

}

/**
  * @brief ADC3 Initialization Function
  * @param None
  * @retval None
  */
static void MX_ADC3_Init(void)
{

  /* USER CODE BEGIN ADC3_Init 0 */

  /* USER CODE END ADC3_Init 0 */

  ADC_ChannelConfTypeDef sConfig = {0};

  /* USER CODE BEGIN ADC3_Init 1 */

  /* USER CODE END ADC3_Init 1 */
  /** Common config 
  */
  hadc3.Instance = ADC3;
  hadc3.Init.ClockPrescaler = ADC_CLOCK_ASYNC_DIV1;
  hadc3.Init.Resolution = ADC_RESOLUTION_16B;
  hadc3.Init.ScanConvMode = ADC_SCAN_DISABLE;
  hadc3.Init.EOCSelection = ADC_EOC_SINGLE_CONV;
  hadc3.Init.LowPowerAutoWait = DISABLE;
  hadc3.Init.ContinuousConvMode = DISABLE;
  hadc3.Init.NbrOfConversion = 1;
  hadc3.Init.DiscontinuousConvMode = ENABLE;
  hadc3.Init.NbrOfDiscConversion = 1;
  hadc3.Init.ExternalTrigConv = ADC_SOFTWARE_START;
  hadc3.Init.ExternalTrigConvEdge = ADC_EXTERNALTRIGCONVEDGE_NONE;
  hadc3.Init.ConversionDataManagement = ADC_CONVERSIONDATA_DR;
  hadc3.Init.Overrun = ADC_OVR_DATA_PRESERVED;
  hadc3.Init.LeftBitShift = ADC_LEFTBITSHIFT_NONE;
  hadc3.Init.OversamplingMode = ENABLE;
  hadc3.Init.Oversampling.Ratio = 8;
  hadc3.Init.Oversampling.RightBitShift = ADC_RIGHTBITSHIFT_3;
  hadc3.Init.Oversampling.TriggeredMode = ADC_TRIGGEREDMODE_SINGLE_TRIGGER;
  hadc3.Init.Oversampling.OversamplingStopReset = ADC_REGOVERSAMPLING_CONTINUED_MODE;
  if (HAL_ADC_Init(&hadc3) != HAL_OK)
  {
    Error_Handler();
  }
  /** Configure Regular Channel 
  */
  sConfig.Channel = ADC_CHANNEL_13;
  sConfig.Rank = ADC_REGULAR_RANK_1;
  sConfig.SamplingTime = ADC_SAMPLETIME_8CYCLES_5;
  sConfig.SingleDiff = ADC_SINGLE_ENDED;
  sConfig.OffsetNumber = ADC_OFFSET_NONE;
  sConfig.Offset = 0;
  if (HAL_ADC_ConfigChannel(&hadc3, &sConfig) != HAL_OK)
  {
    Error_Handler();
  }
  /* USER CODE BEGIN ADC3_Init 2 */


  /* USER CODE END ADC3_Init 2 */

}

/**
  * @brief I2C1 Initialization Function
  * @param None
  * @retval None
  */
static void MX_I2C1_Init(void)
{

  /* USER CODE BEGIN I2C1_Init 0 */

  /* USER CODE END I2C1_Init 0 */

  /* USER CODE BEGIN I2C1_Init 1 */

  /* USER CODE END I2C1_Init 1 */
  hi2c1.Instance = I2C1;
  hi2c1.Init.Timing = 0x00501D55;
  hi2c1.Init.OwnAddress1 = 8;
  hi2c1.Init.AddressingMode = I2C_ADDRESSINGMODE_7BIT;
  hi2c1.Init.DualAddressMode = I2C_DUALADDRESS_DISABLE;
  hi2c1.Init.OwnAddress2 = 0;
  hi2c1.Init.OwnAddress2Masks = I2C_OA2_NOMASK;
  hi2c1.Init.GeneralCallMode = I2C_GENERALCALL_DISABLE;
  hi2c1.Init.NoStretchMode = I2C_NOSTRETCH_ENABLE;
  if (HAL_I2C_Init(&hi2c1) != HAL_OK)
  {
    Error_Handler();
  }
  /** Configure Analogue filter 
  */
  if (HAL_I2CEx_ConfigAnalogFilter(&hi2c1, I2C_ANALOGFILTER_DISABLE) != HAL_OK)
  {
    Error_Handler();
  }
  /** Configure Digital filter 
  */
  if (HAL_I2CEx_ConfigDigitalFilter(&hi2c1, 0) != HAL_OK)
  {
    Error_Handler();
  }
  /** I2C Enable Fast Mode Plus 
  */
  HAL_I2CEx_EnableFastModePlus(I2C_FASTMODEPLUS_I2C1);
  /* USER CODE BEGIN I2C1_Init 2 */

  /* USER CODE END I2C1_Init 2 */

}

/**
  * @brief TIM7 Initialization Function
  * @param None
  * @retval None
  */
static void MX_TIM7_Init(void)
{

  /* USER CODE BEGIN TIM7_Init 0 */

  /* USER CODE END TIM7_Init 0 */

  TIM_MasterConfigTypeDef sMasterConfig = {0};

  /* USER CODE BEGIN TIM7_Init 1 */

  /* USER CODE END TIM7_Init 1 */
  htim7.Instance = TIM7;
  htim7.Init.Prescaler = 0;
  htim7.Init.CounterMode = TIM_COUNTERMODE_UP;
  htim7.Init.Period = 0xFFFF;
  htim7.Init.AutoReloadPreload = TIM_AUTORELOAD_PRELOAD_DISABLE;
  if (HAL_TIM_Base_Init(&htim7) != HAL_OK)
  {
    Error_Handler();
  }
  sMasterConfig.MasterOutputTrigger = TIM_TRGO_RESET;
  sMasterConfig.MasterSlaveMode = TIM_MASTERSLAVEMODE_DISABLE;
  if (HAL_TIMEx_MasterConfigSynchronization(&htim7, &sMasterConfig) != HAL_OK)
  {
    Error_Handler();
  }
  /* USER CODE BEGIN TIM7_Init 2 */

  /* USER CODE END TIM7_Init 2 */

}

/**
  * @brief GPIO Initialization Function
  * @param None
  * @retval None
  */
static void MX_GPIO_Init(void)
{
  GPIO_InitTypeDef GPIO_InitStruct = {0};

  /* GPIO Ports Clock Enable */
  __HAL_RCC_GPIOE_CLK_ENABLE();
  __HAL_RCC_GPIOI_CLK_ENABLE();
  __HAL_RCC_GPIOC_CLK_ENABLE();
  __HAL_RCC_GPIOF_CLK_ENABLE();
  __HAL_RCC_GPIOH_CLK_ENABLE();
  __HAL_RCC_GPIOA_CLK_ENABLE();
  __HAL_RCC_GPIOB_CLK_ENABLE();
  __HAL_RCC_GPIOG_CLK_ENABLE();
  __HAL_RCC_GPIOD_CLK_ENABLE();

  /*Configure GPIO pin Output Level */
  HAL_GPIO_WritePin(GPIOA, GPIO_PIN_5|GPIO_PIN_6|GPIO_PIN_7, GPIO_PIN_RESET);

  /*Configure GPIO pin Output Level */
  HAL_GPIO_WritePin(GPIOC, GPIO_PIN_5|GPIO_PIN_10, GPIO_PIN_RESET);

  /*Configure GPIO pin Output Level */
  HAL_GPIO_WritePin(GPIOB, GPIO_PIN_0|GPIO_PIN_1|GPIO_PIN_2, GPIO_PIN_RESET);

  /*Configure GPIO pins : PE2 PE3 PE4 PE5 
                           PE6 PE7 PE8 PE9 
                           PE10 PE11 PE12 PE13 
                           PE14 PE15 PE0 PE1 */
  GPIO_InitStruct.Pin = GPIO_PIN_2|GPIO_PIN_3|GPIO_PIN_4|GPIO_PIN_5 
                          |GPIO_PIN_6|GPIO_PIN_7|GPIO_PIN_8|GPIO_PIN_9 
                          |GPIO_PIN_10|GPIO_PIN_11|GPIO_PIN_12|GPIO_PIN_13 
                          |GPIO_PIN_14|GPIO_PIN_15|GPIO_PIN_0|GPIO_PIN_1;
  GPIO_InitStruct.Mode = GPIO_MODE_INPUT;
  GPIO_InitStruct.Pull = GPIO_NOPULL;
  HAL_GPIO_Init(GPIOE, &GPIO_InitStruct);

  /*Configure GPIO pins : PI8 PI9 PI10 PI11 
                           PI0 PI1 PI2 PI3 
                           PI4 PI5 PI6 PI7 */
  GPIO_InitStruct.Pin = GPIO_PIN_8|GPIO_PIN_9|GPIO_PIN_10|GPIO_PIN_11 
                          |GPIO_PIN_0|GPIO_PIN_1|GPIO_PIN_2|GPIO_PIN_3 
                          |GPIO_PIN_4|GPIO_PIN_5|GPIO_PIN_6|GPIO_PIN_7;
  GPIO_InitStruct.Mode = GPIO_MODE_INPUT;
  GPIO_InitStruct.Pull = GPIO_NOPULL;
  HAL_GPIO_Init(GPIOI, &GPIO_InitStruct);

  /*Configure GPIO pins : PC13 PC14 PC15 PC0 
                           PC1 PC2 PC3 PC4 
                           PC6 PC7 PC8 PC9 
                           PC11 PC12 */
  GPIO_InitStruct.Pin = GPIO_PIN_13|GPIO_PIN_14|GPIO_PIN_15|GPIO_PIN_0 
                          |GPIO_PIN_1|GPIO_PIN_2|GPIO_PIN_3|GPIO_PIN_4 
                          |GPIO_PIN_6|GPIO_PIN_7|GPIO_PIN_8|GPIO_PIN_9 
                          |GPIO_PIN_11|GPIO_PIN_12;
  GPIO_InitStruct.Mode = GPIO_MODE_INPUT;
  GPIO_InitStruct.Pull = GPIO_NOPULL;
  HAL_GPIO_Init(GPIOC, &GPIO_InitStruct);

  /*Configure GPIO pins : PF0 PF1 PF2 PF3 
                           PF4 PF5 PF6 PF7 
                           PF8 PF9 PF10 PF11 
                           PF12 PF13 PF14 PF15 */
  GPIO_InitStruct.Pin = GPIO_PIN_0|GPIO_PIN_1|GPIO_PIN_2|GPIO_PIN_3 
                          |GPIO_PIN_4|GPIO_PIN_5|GPIO_PIN_6|GPIO_PIN_7 
                          |GPIO_PIN_8|GPIO_PIN_9|GPIO_PIN_10|GPIO_PIN_11 
                          |GPIO_PIN_12|GPIO_PIN_13|GPIO_PIN_14|GPIO_PIN_15;
  GPIO_InitStruct.Mode = GPIO_MODE_INPUT;
  GPIO_InitStruct.Pull = GPIO_NOPULL;
  HAL_GPIO_Init(GPIOF, &GPIO_InitStruct);

  /*Configure GPIO pin : PA5 */
  GPIO_InitStruct.Pin = GPIO_PIN_5;
  GPIO_InitStruct.Mode = GPIO_MODE_OUTPUT_PP;
  GPIO_InitStruct.Pull = GPIO_NOPULL;
  GPIO_InitStruct.Speed = GPIO_SPEED_FREQ_LOW;
  HAL_GPIO_Init(GPIOA, &GPIO_InitStruct);

  /*Configure GPIO pins : PA6 PA7 */
  GPIO_InitStruct.Pin = GPIO_PIN_6|GPIO_PIN_7;
  GPIO_InitStruct.Mode = GPIO_MODE_OUTPUT_PP;
  GPIO_InitStruct.Pull = GPIO_NOPULL;
  GPIO_InitStruct.Speed = GPIO_SPEED_FREQ_VERY_HIGH;
  HAL_GPIO_Init(GPIOA, &GPIO_InitStruct);

  /*Configure GPIO pins : PC5 PC10 */
  GPIO_InitStruct.Pin = GPIO_PIN_5|GPIO_PIN_10;
  GPIO_InitStruct.Mode = GPIO_MODE_OUTPUT_PP;
  GPIO_InitStruct.Pull = GPIO_NOPULL;
  GPIO_InitStruct.Speed = GPIO_SPEED_FREQ_VERY_HIGH;
  HAL_GPIO_Init(GPIOC, &GPIO_InitStruct);

  /*Configure GPIO pins : PB0 PB2 */
  GPIO_InitStruct.Pin = GPIO_PIN_0|GPIO_PIN_2;
  GPIO_InitStruct.Mode = GPIO_MODE_OUTPUT_PP;
  GPIO_InitStruct.Pull = GPIO_NOPULL;
  GPIO_InitStruct.Speed = GPIO_SPEED_FREQ_VERY_HIGH;
  HAL_GPIO_Init(GPIOB, &GPIO_InitStruct);

  /*Configure GPIO pin : PB1 */
  GPIO_InitStruct.Pin = GPIO_PIN_1;
  GPIO_InitStruct.Mode = GPIO_MODE_OUTPUT_PP;
  GPIO_InitStruct.Pull = GPIO_NOPULL;
  GPIO_InitStruct.Speed = GPIO_SPEED_FREQ_LOW;
  HAL_GPIO_Init(GPIOB, &GPIO_InitStruct);

  /*Configure GPIO pins : PG0 PG1 PG2 PG3 
                           PG4 PG5 PG6 PG7 
                           PG8 PG9 PG10 PG11 
                           PG12 PG13 PG14 PG15 */
  GPIO_InitStruct.Pin = GPIO_PIN_0|GPIO_PIN_1|GPIO_PIN_2|GPIO_PIN_3 
                          |GPIO_PIN_4|GPIO_PIN_5|GPIO_PIN_6|GPIO_PIN_7 
                          |GPIO_PIN_8|GPIO_PIN_9|GPIO_PIN_10|GPIO_PIN_11 
                          |GPIO_PIN_12|GPIO_PIN_13|GPIO_PIN_14|GPIO_PIN_15;
  GPIO_InitStruct.Mode = GPIO_MODE_INPUT;
  GPIO_InitStruct.Pull = GPIO_NOPULL;
  HAL_GPIO_Init(GPIOG, &GPIO_InitStruct);

  /*Configure GPIO pins : PB10 PB11 PB12 PB13 
                           PB14 PB15 PB4 PB5 
                           PB8 PB9 */
  GPIO_InitStruct.Pin = GPIO_PIN_10|GPIO_PIN_11|GPIO_PIN_12|GPIO_PIN_13 
                          |GPIO_PIN_14|GPIO_PIN_15|GPIO_PIN_4|GPIO_PIN_5 
                          |GPIO_PIN_8|GPIO_PIN_9;
  GPIO_InitStruct.Mode = GPIO_MODE_INPUT;
  GPIO_InitStruct.Pull = GPIO_NOPULL;
  HAL_GPIO_Init(GPIOB, &GPIO_InitStruct);

  /*Configure GPIO pins : PH6 PH7 PH8 PH9 
                           PH10 PH11 PH12 PH13 
                           PH14 PH15 */
  GPIO_InitStruct.Pin = GPIO_PIN_6|GPIO_PIN_7|GPIO_PIN_8|GPIO_PIN_9 
                          |GPIO_PIN_10|GPIO_PIN_11|GPIO_PIN_12|GPIO_PIN_13 
                          |GPIO_PIN_14|GPIO_PIN_15;
  GPIO_InitStruct.Mode = GPIO_MODE_INPUT;
  GPIO_InitStruct.Pull = GPIO_NOPULL;
  HAL_GPIO_Init(GPIOH, &GPIO_InitStruct);

  /*Configure GPIO pins : PD8 PD9 PD10 PD11 
                           PD12 PD13 PD14 PD15 
                           PD0 PD1 PD2 PD3 
                           PD4 PD5 PD6 PD7 */
  GPIO_InitStruct.Pin = GPIO_PIN_8|GPIO_PIN_9|GPIO_PIN_10|GPIO_PIN_11 
                          |GPIO_PIN_12|GPIO_PIN_13|GPIO_PIN_14|GPIO_PIN_15 
                          |GPIO_PIN_0|GPIO_PIN_1|GPIO_PIN_2|GPIO_PIN_3 
                          |GPIO_PIN_4|GPIO_PIN_5|GPIO_PIN_6|GPIO_PIN_7;
  GPIO_InitStruct.Mode = GPIO_MODE_INPUT;
  GPIO_InitStruct.Pull = GPIO_NOPULL;
  HAL_GPIO_Init(GPIOD, &GPIO_InitStruct);

  /*Configure GPIO pins : PA8 PA9 PA10 PA11 
                           PA12 PA15 */
  GPIO_InitStruct.Pin = GPIO_PIN_8|GPIO_PIN_9|GPIO_PIN_10|GPIO_PIN_11 
                          |GPIO_PIN_12|GPIO_PIN_15;
  GPIO_InitStruct.Mode = GPIO_MODE_INPUT;
  GPIO_InitStruct.Pull = GPIO_NOPULL;
  HAL_GPIO_Init(GPIOA, &GPIO_InitStruct);

}

/* USER CODE BEGIN 4 */
void I2C1_ClearBusyFlagErratum(I2C_HandleTypeDef *instance)
{
    GPIO_InitTypeDef GPIO_InitStruct;
    int timeout =100;
    int timeout_cnt=0;

    // 1. Clear PE bit.
    instance->Instance->CR1 &= ~(0x0001);

    //  2. Configure the SCL and SDA I/Os as General Purpose Output Open-Drain, High level (Write 1 to GPIOx_ODR).
    GPIO_InitStruct.Mode         = GPIO_MODE_OUTPUT_OD;
    GPIO_InitStruct.Alternate    = GPIO_AF4_I2C1;
    GPIO_InitStruct.Pull         = GPIO_PULLUP;
    GPIO_InitStruct.Speed        = GPIO_SPEED_FREQ_HIGH;

    GPIO_InitStruct.Pin          = I2C1_SCL_PIN;
    HAL_GPIO_Init(I2C1_SCL_PORT, &GPIO_InitStruct);
    HAL_GPIO_WritePin(I2C1_SCL_PORT, I2C1_SCL_PIN, GPIO_PIN_SET);

    GPIO_InitStruct.Pin          = I2C1_SDA_PIN;
    HAL_GPIO_Init(I2C1_SDA_PORT, &GPIO_InitStruct);
    HAL_GPIO_WritePin(I2C1_SDA_PORT, I2C1_SDA_PIN, GPIO_PIN_SET);


    // 3. Check SCL and SDA High level in GPIOx_IDR.
    while (GPIO_PIN_SET != HAL_GPIO_ReadPin(I2C1_SCL_PORT, I2C1_SCL_PIN))
    {
        timeout_cnt++;
        if(timeout_cnt>timeout)
            return;
    }

    while (GPIO_PIN_SET != HAL_GPIO_ReadPin(I2C1_SDA_PORT, I2C1_SDA_PIN))
    {
        //Move clock to release I2C
        HAL_GPIO_WritePin(I2C1_SCL_PORT, I2C1_SCL_PIN, GPIO_PIN_RESET);
        asm("nop");
        HAL_GPIO_WritePin(I2C1_SCL_PORT, I2C1_SCL_PIN, GPIO_PIN_SET);

        timeout_cnt++;
        if(timeout_cnt>timeout)
            return;
    }

    // 4. Configure the SDA I/O as General Purpose Output Open-Drain, Low level (Write 0 to GPIOx_ODR).
    HAL_GPIO_WritePin(I2C1_SDA_PORT, I2C1_SDA_PIN, GPIO_PIN_RESET);

    //  5. Check SDA Low level in GPIOx_IDR.
    while (GPIO_PIN_RESET != HAL_GPIO_ReadPin(I2C1_SDA_PORT, I2C1_SDA_PIN))
    {
        timeout_cnt++;
        if(timeout_cnt>timeout)
            return;
    }

    // 6. Configure the SCL I/O as General Purpose Output Open-Drain, Low level (Write 0 to GPIOx_ODR).
    HAL_GPIO_WritePin(I2C1_SCL_PORT, I2C1_SCL_PIN, GPIO_PIN_RESET);

    //  7. Check SCL Low level in GPIOx_IDR.
    while (GPIO_PIN_RESET != HAL_GPIO_ReadPin(I2C1_SCL_PORT, I2C1_SCL_PIN))
    {
        timeout_cnt++;
        if(timeout_cnt>timeout)
            return;
    }

    // 8. Configure the SCL I/O as General Purpose Output Open-Drain, High level (Write 1 to GPIOx_ODR).
    HAL_GPIO_WritePin(I2C1_SCL_PORT, I2C1_SCL_PIN, GPIO_PIN_SET);

    // 9. Check SCL High level in GPIOx_IDR.
    while (GPIO_PIN_SET != HAL_GPIO_ReadPin(I2C1_SCL_PORT, I2C1_SCL_PIN))
    {
        timeout_cnt++;
        if(timeout_cnt>timeout)
            return;
    }

    // 10. Configure the SDA I/O as General Purpose Output Open-Drain , High level (Write 1 to GPIOx_ODR).
    HAL_GPIO_WritePin(I2C1_SDA_PORT, I2C1_SDA_PIN, GPIO_PIN_SET);

    // 11. Check SDA High level in GPIOx_IDR.
    while (GPIO_PIN_SET != HAL_GPIO_ReadPin(I2C1_SDA_PORT, I2C1_SDA_PIN))
    {
        timeout_cnt++;
        if(timeout_cnt>timeout)
            return;
    }

    // 12. Configure the SCL and SDA I/Os as Alternate function Open-Drain.
    GPIO_InitStruct.Mode = GPIO_MODE_AF_OD;
    GPIO_InitStruct.Pull = GPIO_PULLUP;
    GPIO_InitStruct.Speed = GPIO_SPEED_FREQ_LOW;
    GPIO_InitStruct.Alternate = GPIO_AF4_I2C1;

    GPIO_InitStruct.Pin = I2C1_SCL_PIN;
    HAL_GPIO_Init(I2C1_SCL_PORT, &GPIO_InitStruct);

    GPIO_InitStruct.Pin = I2C1_SDA_PIN;
    HAL_GPIO_Init(I2C1_SDA_PORT, &GPIO_InitStruct);

    HAL_GPIO_WritePin(I2C1_SCL_PORT, I2C1_SCL_PIN, GPIO_PIN_SET);
    HAL_GPIO_WritePin(I2C1_SDA_PORT, I2C1_SDA_PIN, GPIO_PIN_SET);

    // 13. Set SWRST bit in I2Cx_CR1 register.
    instance->Instance->CR1 |= 0x8000;

    asm("nop");

    // 14. Clear SWRST bit in I2Cx_CR1 register.
    instance->Instance->CR1 &= ~0x8000;

    asm("nop");

    // 15. Enable the I2C peripheral by setting the PE bit in I2Cx_CR1 register
    instance->Instance->CR1 |= 0x0001;

    // Call initialization function.
    HAL_I2C_Init(instance);
}
/* USER CODE END 4 */

/**
  * @brief  This function is executed in case of error occurrence.
  * @retval None
  */
void Error_Handler(void)
{
  /* USER CODE BEGIN Error_Handler_Debug */
	/* User can add his own implementation to report the HAL error return state */

  /* USER CODE END Error_Handler_Debug */
}

#ifdef  USE_FULL_ASSERT
/**
  * @brief  Reports the name of the source file and the source line number
  *         where the assert_param error has occurred.
  * @param  file: pointer to the source file name
  * @param  line: assert_param error line source number
  * @retval None
  */
void assert_failed(uint8_t *file, uint32_t line)
{ 
  /* USER CODE BEGIN 6 */
	/* User can add his own implementation to report the file name and line number,
     tex: printf("Wrong parameters value: file %s on line %d\r\n", file, line) */
  /* USER CODE END 6 */
}
#endif /* USE_FULL_ASSERT */

/************************ (C) COPYRIGHT STMicroelectronics *****END OF FILE****/
