/*
    Onboard software for the AMDV
    Developer:  Bob Augustine
    Dates of Development:  8.2.09
    Version:  12.0
*/

//Nonstandard Libraries In Use
#include <NewSoftSerial.h>
#include "TinyGPS.h"
#include <stdlib.h>

/*
    For Testing When Metal Detector Is Not Connected
    true = Metal Detector Connected
    false = Metal Detector Not Connected
    When not connected, then connect line from Digital pin 3 to analog pin 1
*/
#include "WProgram.h"
void setup();
void loop();
void reset_AMDV(boolean firstTimeThrough);
void initialize_Pins();
void initialize_LEDs();
void initialize_Varibles();
void  initialize_Bluetooth();
void initialize_Metal_Detector();
void initialize_GPS();
void confirmReady();
void checkConnection();
void process_Bluetooth();
void confirmCommand(int commandToConfirm);
void do_Manual();
void do_Auto();
void checkObstructions();
void setMotors();
void motorsOff();
void checkMetal();
void do_GPS();
void send_GPS();
void get_GPS();
boolean metal_detector_connected = true;


//Movement and Mode and Alert Definitions
#define AUTOMODE '7'
#define MANUALMODE '8'
#define FORWARD '1'
#define FORWARDLEFT '2'
#define FORWARDRIGHT '3'
#define REVERSE '4'
#define REVERSELEFT '5'
#define REVERSERIGHT '6'
#define STOP '0'
#define METALFOUND 'A'
#define METALNOTFOUND 'B'
#define OBSTRUCTIONFOUND 'C'
#define AMDVREADY 'J'
#define GPSREADY 'G'
#define GPSNOFIX 'H'
#define BLUETOOTHREADY 'I'
#define USEGPSOFF 'O'
#define CONTINUESEARCH 'S'
#define GPSNOTCONNECTED 'K'
#define AMDVPING 'Z'
#define METALNOGPS 'P'
#define GPSBEGIN '$'
#define GPSLAT '*'
#define GPSLONG '^'
#define AUTOMODEOVER 'L'
#define METALDETECTORSET 'M'
#define METALAVERAGERECIEVED 'N'

//Threshold Definitions
//#define METALTHRESH 217//100 //Metal detector threshold, less than value = metal found
#define SENSORTHRESH 45 //Sensor threshold, less than value = obstruction found
#define GPS_WAIT_TIME 5000//20000 //Wait time for getting valid GPS fix
#define RESEND_WAIT_TIME 500 //Wait time to resend commands

//Pin Definitions
#define METAL_DETECTOR_PIN 1 //Analog pin 1
#define SENSOR_PIN 0 //Analog pin 0
#define GPS_RX_PIN 4  //Digital Pin 4

#define GPS_TX_PIN 255 //Not a real pin - Not being used
#define DRIVE_MOTOR_FORWARD_PIN 5 //Digital Pin 5
#define DRIVE_MOTOR_REVERSE_PIN 6 //Digital Pin 6
#define DRIVE_MOTOR_ENABLE_PIN 2 //Digital Pin 2
#define STEERING_MOTOR_RIGHT_PIN 7 //Digital Pin 7
#define STEERING_MOTOR_LEFT_PIN 8 //Digital Pin 8
#define LED_AMDV_ON 9 //Digital Pin 9
#define LED_AMDV_READY 10 //Digital Pin 10
#define LED_GPS_ON 11 //Digital Pin 11
#define LED_METAL_FOUND 12 //Digital Pin 12
#define LED_OBSTRUCTION_FOUND 13 //Digital Pin 13
#define FAKEMETALPIN 3 //Digital Pin 3 used when metal detector not connected

//Variable Declerations
int mode; //Which mode AMDV is in
int moveDirection; //Which direction to move
int previousMoveDirection;  //Previous direction which AMDV was moving
boolean GPS_Online; //If GPS is to be used
TinyGPS gps; //GPS object
long lat, lon; //Latitude and Longitude from GPS
unsigned long fix_age; //Fix age from GPS
NewSoftSerial gpsSerial(GPS_RX_PIN,GPS_TX_PIN); //New softserial object for GPS (GPS TX,GPS RX)
boolean metalFoundAlready; //This metal was already found
unsigned long tempTime; //Temporary time variable
unsigned long timeToCheckConnection; //When to check the connection
int metalThresh; //Used to set the metal thresh automatically based on if detector is connected or not

//Auto mode variables
int maxCircleCount = 3; //Maximum allowed number of loops
int autoModeCircleCount = maxCircleCount; //Used to keep track of the number of drive loops AMDV has done in auto mode
int autoModeIndex = 0; //Used to keep index of location for auto operation
unsigned long autoModeTimeToNext = 0;
boolean metalDetectorDisabled;
unsigned long metalDisableTime=0;
boolean obstructionFound;
unsigned long obstructionTime;

/*unsigned long stopMetalTime;
boolean stopMetal;*/

//Used for testing metal detector when not connected
unsigned long metalTime;
int state;

//Required Function: Runs once on power-on
void setup(){
     
    reset_AMDV(true);    

}

//Required Function: Continually loops until power off
void loop(){
  
    /*if(GPS_Online==false)
        GPS_Online=true;
    */    
    if(metal_detector_connected==false && metalTime < millis()){
        state=!state;
        digitalWrite(FAKEMETALPIN,state);
        metalTime=millis()+30000;
    }
    
    if(timeToCheckConnection<millis()){
        checkConnection();
    }
    //Update state of AMDV by reading from bluetooth
    while(Serial.available()>0){
        process_Bluetooth();
    }
    //Check which mode AMDV is in and follow that program path
    if(mode==AUTOMODE){
        do_Auto();
    }
    else{//mode==MANUALMODE
        do_Manual();
    } 
}

/******************************************
 *  Initial Setup Functions               *
 *  Gets the AMDV to a Ready State        *
 ******************************************/
 
//Resets the AMDV to an initial state
void reset_AMDV(boolean firstTimeThrough){
    
    //Metal Detector Initialization
    //metalThresh=METALTHRESH;
    if(metal_detector_connected==false){
        metalThresh = 500;
        state = LOW;
        pinMode(FAKEMETALPIN,OUTPUT);
        digitalWrite(FAKEMETALPIN,state);
    }
    
    //Set Pin Operation Modes and Initial States
    initialize_Pins();  
    
    //Initialize all variables
    initialize_Varibles();
    
    //Setup and Initialize Bluetooth
    initialize_Bluetooth();
    
    //Setup and Initialize Metal Detector
    initialize_Metal_Detector();
    
    //Setup and Initialize GPS
    initialize_GPS();
    
    delay(500);
    
    //Confirm AMDV ready
    confirmReady();
    metalTime=millis()+15000;
    if(firstTimeThrough==false){
        loop();
    }
    
}

//Initialize pin states
void initialize_Pins(){
    /**********************************************************************************
      Add pin modes and initial states
    ***********************************************************************************/
    pinMode(LED_AMDV_ON,OUTPUT);
    pinMode(LED_AMDV_READY,OUTPUT);
    pinMode(LED_GPS_ON,OUTPUT);
    pinMode(LED_METAL_FOUND,OUTPUT);
    pinMode(LED_OBSTRUCTION_FOUND,OUTPUT);

    pinMode(DRIVE_MOTOR_ENABLE_PIN,OUTPUT);
    digitalWrite(DRIVE_MOTOR_ENABLE_PIN,LOW);
    pinMode(DRIVE_MOTOR_FORWARD_PIN,OUTPUT);
    digitalWrite(DRIVE_MOTOR_FORWARD_PIN,LOW);
    pinMode(DRIVE_MOTOR_REVERSE_PIN,OUTPUT);
    digitalWrite(DRIVE_MOTOR_REVERSE_PIN,LOW);
    pinMode(STEERING_MOTOR_RIGHT_PIN,OUTPUT);
    digitalWrite(STEERING_MOTOR_RIGHT_PIN,LOW);
    pinMode(STEERING_MOTOR_LEFT_PIN,OUTPUT);
    digitalWrite(STEERING_MOTOR_LEFT_PIN,LOW);
    
    //Set initial LED States
    initialize_LEDs();
}

//Set inital LED States
void initialize_LEDs(){
    digitalWrite(LED_AMDV_ON,HIGH);
    digitalWrite(LED_AMDV_READY,LOW);
    digitalWrite(LED_GPS_ON,HIGH);
    digitalWrite(LED_METAL_FOUND,LOW);
    digitalWrite(LED_OBSTRUCTION_FOUND,LOW);
}

//Initialize Variables
void initialize_Varibles(){    
    mode = MANUALMODE;
    moveDirection = STOP;
    previousMoveDirection = -1;//Set to not a valid direction//FORWARD;
    GPS_Online = true;
    metalFoundAlready = false;
    timeToCheckConnection=0;
    autoModeCircleCount = maxCircleCount;
    autoModeIndex = 0;
    metalDetectorDisabled=false;
    obstructionFound=false;
   // stopMetal=false;
}

//Setup and Initialize Bluetooth
void  initialize_Bluetooth(){
    Serial.begin(115200);
    delay(500);    
                        
    //Check if Bluetooth Ready
    tempTime=millis()+RESEND_WAIT_TIME;//Wait
    Serial.print(BLUETOOTHREADY,BYTE);
    while(true){
      if(Serial.available()>0){
          if(BLUETOOTHREADY==Serial.read()){
              break;
          }
      }
      if(tempTime<millis()){
          Serial.print(BLUETOOTHREADY,BYTE);
          tempTime=millis()+RESEND_WAIT_TIME;//Wait
      }
    } 
}

//Setup and Initialize Metal Detector
void initialize_Metal_Detector(){
  int maxFailures;
  //Set stream for metal detector readings
  maxFailures=6;
    tempTime=millis()+RESEND_WAIT_TIME;//Wait
    Serial.print(METALDETECTORSET,BYTE);
    while(true){
     if(Serial.available()>0){
          if(METALDETECTORSET==Serial.read()){
              break;
          }
      }
      if(tempTime<millis()){
          Serial.print(METALDETECTORSET,BYTE);
          tempTime=millis()+RESEND_WAIT_TIME;//Wait
          maxFailures=maxFailures-1;
          if(maxFailures<0)
              reset_AMDV(false);
      }
    }
    //Send values to computer
    while(true){
      
      int val=0;
      int sum=0;
      for(int i=0;i<5;i++){
         delay(100);
         val=analogRead(METAL_DETECTOR_PIN);
         sum=sum+val;
         Serial.print(val);
         Serial.print(' ');
         Serial.print(' ');
      }
      /*Serial.print(' ');
      Serial.print(' ');
      Serial.print(' ');
      Serial.print('A');
      Serial.print('V');
      Serial.print('E');
      Serial.print('R');
      Serial.print('A');
      Serial.print('G');
      Serial.print('E');
      Serial.print(':');
      Serial.print(' ');*/
      Serial.print(METALAVERAGERECIEVED,BYTE);
      
      Serial.println(sum/5);
      if(Serial.available()>0){
           //if(METALDETECTORSET==Serial.read()){
             //metalThresh=1000;
                 char buffer[6];
                 for(int i=0;i<4;i++){
                    while(Serial.available()<=0){
                    }
                    buffer[i]=Serial.read();
                 }
                 buffer[5]='\0';
                 delay(500);
                 metalThresh=atoi(buffer);
                 if(metalThresh>1022)
                     metalThresh=1022;
                 if(metalThresh<0)
                     metalThresh=0;
                 break;
           //}
      }
    }
    //Send close stream
    tempTime=millis()+RESEND_WAIT_TIME;//Wait
    Serial.print(METALDETECTORSET,BYTE);
    maxFailures=6;
    while(true){
      if(Serial.available()>0){
          if(METALDETECTORSET==Serial.read()){
              break;
          }
       }
       if(tempTime<millis()){
          Serial.print(METALDETECTORSET,BYTE);
          tempTime=millis()+RESEND_WAIT_TIME;//Wait
          maxFailures=maxFailures-1;
          if(maxFailures<0)
              reset_AMDV(false);
       }
    }
    delay(500);
    return;
}

//Setup and Initialize GPS
void initialize_GPS(){
    
    unsigned long toExitLoopTime;
    boolean tooMuchTimeElapsed=false;
    int flag2=0;//Assume GPS is not connected
    int maxFailures=3;
    
    
    digitalWrite(LED_GPS_ON,HIGH);
    gpsSerial.begin(4800);
    delay(500);
     
    toExitLoopTime=millis()+GPS_WAIT_TIME;//time to wait for valid fix

    while(tooMuchTimeElapsed==false){
        while(gpsSerial.available()){
          flag2=1;//GPS is connected
            
            //Get GPS data and process it
          if(gps.encode(gpsSerial.read())==true){
             
            //gps.encode(gpsSerial.read());
            gps.get_position(&lat,&lon,&fix_age);//Get position and fix age

            if(fix_age<1000){
               //Have recieved location from satellites
               tooMuchTimeElapsed=true;
               break;
            }
          }
          else{// if(fix_age==TinyGPS::GPS_INVALID_AGE || fix_age>5000){
               //Have not recieved location from satellites
               if(toExitLoopTime<millis()){
                    
                  //Ask user if want to continue searching or use with GPS off
                  tempTime=millis()+1000;//Wait for three seconds
                  Serial.print(GPSNOFIX,BYTE);
                  int data;
                  int flag=0;
                  while(true){
                      if(Serial.available()>0){
                          data=Serial.read();
                          if(GPSNOFIX==data){
                              flag=1;
                          }
                          else if(AMDVPING==data){
                              maxFailures=3;
                              tempTime=millis()+RESEND_WAIT_TIME;
                          }
                          else if(USEGPSOFF==data){
                              GPS_Online=false;
                              digitalWrite(LED_GPS_ON,LOW);
                              tooMuchTimeElapsed=true;
                              break;
                          }
                          else if(CONTINUESEARCH==data){
                               flag=0;
                               toExitLoopTime=millis()+GPS_WAIT_TIME;
                               break;
                          }
                      }
                      if(tempTime<millis() && flag==0){
                          //No Fix Not Confirmed
                          Serial.print(GPSNOFIX,BYTE);
                          tempTime=millis()+RESEND_WAIT_TIME;//Wait
                          maxFailures--;
                          if(maxFailures<=0){
                              reset_AMDV(false);//Reset the board
                              //setup();//Reset the board
                          }
                      }
                      else if(tempTime<millis() && flag==1){
                          //No Fix Confirmed, Waiting on input
                          Serial.print(AMDVPING,BYTE);
                          maxFailures=maxFailures-1;
                          if(maxFailures<=0){
                              reset_AMDV(false);//Reset the board
                          }                       
                          tempTime=millis()+RESEND_WAIT_TIME;
                      }
                  }  
                  if(tooMuchTimeElapsed==true){ 
                       break;
                  }
               }
          }   
        }
        //GPS Not connected
        if(toExitLoopTime<millis() && flag2==0){
              //Inform User No GPS Connected
              tempTime=millis()+RESEND_WAIT_TIME;//Wait
              Serial.print(GPSNOTCONNECTED,BYTE);
              maxFailures=6;
              while(true){
                  if(Serial.available()>0){
                    if(GPSNOTCONNECTED==Serial.read()){
                        break;
                    }
                  }
                  if(tempTime<millis()){
                      Serial.print(GPSNOTCONNECTED,BYTE);
                      tempTime=millis()+RESEND_WAIT_TIME;//Wait
                      maxFailures--;
                      if(maxFailures<=0){
                          reset_AMDV(false);//Reset the boa 
                          //setup();//Reset the board
                      }
                  }
              }  
          
              GPS_Online=false;
              digitalWrite(LED_GPS_ON,LOW);
              tooMuchTimeElapsed=true;
        }  
    }
    
    //GPS Ready
    maxFailures=3;
    tempTime=millis()+RESEND_WAIT_TIME;//Wait
    Serial.print(GPSREADY,BYTE);
    while(true){
      if(Serial.available()>0){
          if(GPSREADY==Serial.read()){
              break;
          }
      }
      /***********************************/
      // Use ping code here to check to see if connection is still good
      /***********************************/
      
      if(tempTime<millis()){
          Serial.print(GPSREADY,BYTE);
          tempTime=millis()+RESEND_WAIT_TIME;//Wait
          maxFailures--;
          if(maxFailures<=0){
              reset_AMDV(false);//Reset the board
          }
      }
    } 
}

//Alert user AMDV is ready
void confirmReady(){
    //AMDV is ready
    int maxFailures=3;
    //Inform User AMDV ready
    tempTime=millis()+RESEND_WAIT_TIME;//Wait
    Serial.print(AMDVREADY,BYTE);
    while(true){
      if(Serial.available()>0){
          if(AMDVREADY==Serial.read()){
              break;
          }
      }
      if(tempTime<millis()){
          Serial.print(AMDVREADY,BYTE);
          tempTime=millis()+RESEND_WAIT_TIME;//Wait
          maxFailures--;
          if(maxFailures<=0){
               reset_AMDV(false);//Reset the boa
          }
      }
    }  
      
    digitalWrite(LED_AMDV_READY,HIGH);  
}

/******************************************
 *  Running Functions                     *
 *  Handles the running of the AMDV       *
 ******************************************/
 
//Check if still connected
void checkConnection(){
    int maxFailures = 9;
    tempTime=millis()+RESEND_WAIT_TIME;//Wait
    Serial.print(AMDVPING,BYTE);
    while(true){
      if(Serial.available()>0){
          if(AMDVPING==Serial.read()){
              break;
          }
      }
      else if(tempTime<millis()){
          Serial.print(AMDVPING,BYTE);
          tempTime=millis()+RESEND_WAIT_TIME;//Wait
          maxFailures=maxFailures-1;
          if(maxFailures==0){
              reset_AMDV(false);//Reset the board
          }
      }
    }
    
    timeToCheckConnection=millis()+RESEND_WAIT_TIME;//Re check connection
}
 
//Process incoming data from bluetooth
void process_Bluetooth(){
    int dataIn;
    
    //Read a byte from bluetooth
    dataIn=Serial.read();
    
    //Check if was a mode command
    if(dataIn==MANUALMODE || dataIn==AUTOMODE){
        //Set the mode
        mode=dataIn;
        
        //Confirm the recieved command
        confirmCommand(dataIn);
        
        //if auto mode reset automode variables
        if(mode==AUTOMODE){
            autoModeCircleCount = maxCircleCount;
            autoModeIndex = 0;
            previousMoveDirection = -1;
            autoModeTimeToNext = 0;
        }
        if(mode==MANUALMODE){
            moveDirection=STOP;
            setMotors();
            metalDetectorDisabled=false;
            obstructionFound=false;
        }
    }
    //Check if was a movement command
    else if(dataIn>=STOP && dataIn<=REVERSERIGHT){
        //Set the movement direction
        moveDirection=dataIn;
        
        //Confirm the recieved command
        confirmCommand(dataIn);
    }
    //It was another command
    else{
        //Not a recognized command, Do nothing
    }
}

//Confirms that previous command recieved
void confirmCommand(int commandToConfirm){
    Serial.print(commandToConfirm,BYTE);
}
 
//Manual Mode Control Loop
void do_Manual(){
    //Set Motors
    setMotors();
  
    //Check for Metal
    if(metalDetectorDisabled==false){
        checkMetal();
    }

    //Check for obstructions
 //   checkObstructions();
         
  
}

//Auto Mode Control Loop
void do_Auto(){
  
         /**********************************************************************************
           Add auto mode stuff
           
           -Check for metal
           -Check for obstructions
           -Set motors based on correct place in array
           -Update array index and loop count
         ***********************************************************************************/
         
         //If already exceeded max circles allowed, return
         if(autoModeCircleCount<0){
               if(autoModeCircleCount!=-5){
                     autoModeCircleCount=-5;
                     Serial.print(AUTOMODEOVER,BYTE);    
                     mode=MANUALMODE;
                     obstructionFound=false;
               }
               return;
         }
         
         //Check for metal
         if(metalDetectorDisabled==false){
             checkMetal();
         }
         
         //Check for obstructions
         checkObstructions();
         
         //Which auto mode to use
         //3 point turn --> autoType=0
         //Half moon turn --> autoType=1
         int autoType=1;
         
         //If current time is greater than exit time
         if((autoModeTimeToNext < millis()) && (autoModeCircleCount>=0)){
    
           if(autoType==0){
                  //Three Point Turn - Auto Mode 
             
                 //Set the move direction based on index
                  if(autoModeIndex==0){
                      moveDirection=STOP;
                      autoModeTimeToNext=300;
                  }
                  else if(autoModeIndex==1){
                      metalDetectorDisabled=false;
                      moveDirection=FORWARD;
                      autoModeTimeToNext=3000;
                  }
                  else if(autoModeIndex==2){
                      metalDetectorDisabled=true;
                      moveDirection=STOP;
                      autoModeTimeToNext=300;
                  }
                  else if(autoModeIndex==3){
                      moveDirection=FORWARDRIGHT;
                      autoModeTimeToNext=1250;
                  }
                  else if(autoModeIndex==4){
                      moveDirection=REVERSELEFT;
                      autoModeTimeToNext=2250;//2100, 2500
                  }
                  else if(autoModeIndex==5){
                      moveDirection=FORWARDRIGHT;
                      autoModeTimeToNext=1750;
                  }
                  else if(autoModeIndex==6){
                      moveDirection=STOP;
                      autoModeTimeToNext=300;
                  }
                  else if(autoModeIndex==7){
                      metalDetectorDisabled=false;
                      moveDirection=FORWARD;
                      autoModeTimeToNext=3200;//3750,3000
                  }
                  else if(autoModeIndex==8){
                      metalDetectorDisabled=true;
                      moveDirection=STOP;
                      autoModeTimeToNext=300;
                  }
                  else if(autoModeIndex==9){
                      moveDirection=FORWARDLEFT;
                      autoModeTimeToNext=1250;
                  }
                  else if(autoModeIndex==10){
                      moveDirection=REVERSERIGHT;
                      autoModeTimeToNext=2100;//2500
                  }
                  else if(autoModeIndex==11){
                      moveDirection=FORWARDLEFT;
                      autoModeTimeToNext=1750;
                  }
                  else if(autoModeIndex==12){
                      metalDetectorDisabled=false;
                      moveDirection=STOP;
                      autoModeTimeToNext=300;
                  }
                  autoModeTimeToNext=autoModeTimeToNext+millis();
                         
                  //Set the motors
                  setMotors();
                  
                  //Increment index
                  autoModeIndex=autoModeIndex+1;
         
                 //Loop Array Index Around
                 if(12<autoModeIndex){
                     autoModeIndex=0;
                     autoModeCircleCount=autoModeCircleCount-1;         
                 }
                 
           }
           else if(autoType==1){
           
                 //Set the move direction based on index
                  if(autoModeIndex==0){
                      moveDirection=STOP;
                      autoModeTimeToNext=200;
                  }
                  else if(autoModeIndex==1){
                      moveDirection=FORWARD;
                      autoModeTimeToNext=4000;
                      metalDetectorDisabled=false;
                      if(obstructionFound==true){
                           autoModeTimeToNext=autoModeTimeToNext-obstructionTime;
                           obstructionFound=false; 
                      }
                  }
                  else if(autoModeIndex==2){
                      moveDirection=STOP;
                      autoModeTimeToNext=600;
                      metalDetectorDisabled=true;
                  }
                  else if(autoModeIndex==3){
                      moveDirection=FORWARDRIGHT;
                      autoModeTimeToNext=3930;
                  }
                  else if(autoModeIndex==4){
                      moveDirection=STOP;
                      autoModeTimeToNext=600;//2100, 2500
                  }
                  else if(autoModeIndex==5){
                      metalDetectorDisabled=false;
                      moveDirection=FORWARD;
                      autoModeTimeToNext=4000;
                      if(obstructionFound==true){
                           autoModeTimeToNext=autoModeTimeToNext-obstructionTime;
                           obstructionFound=false; 
                      }
                  }
                  else if(autoModeIndex==6){
                      moveDirection=STOP;
                      autoModeTimeToNext=600;
                      metalDetectorDisabled=true;
                  }
                  else if(autoModeIndex==7){
                      moveDirection=FORWARDRIGHT;
                      autoModeTimeToNext=2150;//3750,3000
                  }
                  else if(autoModeIndex==8){
                      moveDirection=FORWARD;
                      autoModeTimeToNext=100;
                  }
                  else if(autoModeIndex==9){
                      moveDirection=FORWARDRIGHT;
                      autoModeTimeToNext=1750;
                  }
                  else if(autoModeIndex==10){
                      moveDirection=STOP;
                      autoModeTimeToNext=500;//2500
                      metalDetectorDisabled=false;
                  }
                  autoModeTimeToNext=autoModeTimeToNext+millis();
                         
                  //Set the motors
                  setMotors();
                  
                  //Increment index
                  autoModeIndex=autoModeIndex+1;
         
                 //Loop Array Index Around
                 if(10<autoModeIndex){
                     autoModeIndex=0;
                     autoModeCircleCount=autoModeCircleCount-1;         
                 }
           
           }
           
       }
         
}

void checkObstructions(){
      //CHECK FOR OBSTRUCTIONS
  
      if(analogRead(SENSOR_PIN)<SENSORTHRESH){    
        //Amount of time left before going to next operation
        obstructionTime=autoModeTimeToNext-millis();
        
        //Stop AMDV and Switch to manual control for the time being
        moveDirection=STOP;
        setMotors();
        
        //Array index where forward commands are
        int forwardLoco1 = 1;
        int forwardLoco2 = 5;
        
        //If in straight away on auto mode
        if(autoModeIndex==(forwardLoco1+1) || autoModeIndex==(forwardLoco1+1)){
            //Go in reverse
            moveDirection=REVERSE;
            setMotors();
            
            //let it back up some
            delay(750);
            
            //Stop
            moveDirection=STOP;
            setMotors();
            
            //Set the time to make the car go into the turn
            autoModeTimeToNext=0;
            obstructionFound=true;            
        }
        //Not in straight away....so unknown obstruction
        else{
            autoModeCircleCount=-1;
          
            int maxFailures = 3;
            Serial.print(OBSTRUCTIONFOUND,BYTE);
            tempTime=millis()+RESEND_WAIT_TIME;//Wait
            while(true){
                if(Serial.available()>0){
                  if(OBSTRUCTIONFOUND==Serial.read()){
                      break;
                  }
                }
                else if(tempTime<millis()){
                  Serial.print(OBSTRUCTIONFOUND,BYTE);
                  tempTime=millis()+RESEND_WAIT_TIME;//Wait
                  maxFailures=maxFailures-1;
                  if(maxFailures==0){
                      checkConnection();
                      maxFailures=3;
                  }
                }
            }  
        }
      }
}


//Set the motors to the correction combination to produce desired results
void setMotors(){
  
    //Check if already moving in same direction
    if(previousMoveDirection==moveDirection){
        previousMoveDirection=moveDirection;
    } 
    //Not same direction so, set motors
    else{
        //Set motors off
        motorsOff();
    
        //Set correct motors on
        if(moveDirection == FORWARD){
            digitalWrite(DRIVE_MOTOR_FORWARD_PIN, HIGH);
            digitalWrite(DRIVE_MOTOR_ENABLE_PIN, HIGH);
        }
        else if(moveDirection == FORWARDLEFT){
            digitalWrite(DRIVE_MOTOR_FORWARD_PIN, HIGH);
            digitalWrite(STEERING_MOTOR_LEFT_PIN, HIGH);
            digitalWrite(DRIVE_MOTOR_ENABLE_PIN, HIGH);
        }
        else if(moveDirection == FORWARDRIGHT){
            digitalWrite(DRIVE_MOTOR_FORWARD_PIN, HIGH);
            digitalWrite(STEERING_MOTOR_RIGHT_PIN, HIGH);
            digitalWrite(DRIVE_MOTOR_ENABLE_PIN, HIGH);
        }
        else if(moveDirection == REVERSE){
            digitalWrite(DRIVE_MOTOR_REVERSE_PIN, HIGH);
            digitalWrite(DRIVE_MOTOR_ENABLE_PIN, HIGH);
        }
        else if(moveDirection == REVERSELEFT){
            digitalWrite(DRIVE_MOTOR_REVERSE_PIN, HIGH);
            digitalWrite(STEERING_MOTOR_LEFT_PIN, HIGH);
            digitalWrite(DRIVE_MOTOR_ENABLE_PIN, HIGH);
        }
        else if(moveDirection == REVERSERIGHT){
            digitalWrite(DRIVE_MOTOR_REVERSE_PIN, HIGH);
            digitalWrite(STEERING_MOTOR_RIGHT_PIN, HIGH);
            digitalWrite(DRIVE_MOTOR_ENABLE_PIN, HIGH);
        }
        else{//moveDirection == STOP
        
            //stopMetalTime=millis();
            //stopMetal = true;
            
            //Use fast stop!
            if(previousMoveDirection==FORWARD || previousMoveDirection==FORWARDRIGHT || previousMoveDirection==FORWARDLEFT){
                //Set into reverse
                digitalWrite(DRIVE_MOTOR_ENABLE_PIN, LOW);
                digitalWrite(DRIVE_MOTOR_FORWARD_PIN, LOW);
                digitalWrite(DRIVE_MOTOR_REVERSE_PIN, HIGH);
                digitalWrite(STEERING_MOTOR_RIGHT_PIN, LOW);
                digitalWrite(STEERING_MOTOR_LEFT_PIN, LOW);
                digitalWrite(DRIVE_MOTOR_ENABLE_PIN, HIGH);
                delay(460);  
                motorsOff();
              //  delay(2000);
            }
            else if(previousMoveDirection==REVERSE || previousMoveDirection==REVERSERIGHT || previousMoveDirection==REVERSELEFT){
                //Set into forward
                digitalWrite(DRIVE_MOTOR_ENABLE_PIN, LOW);
                digitalWrite(DRIVE_MOTOR_FORWARD_PIN, HIGH);
                digitalWrite(DRIVE_MOTOR_REVERSE_PIN, LOW);
                digitalWrite(STEERING_MOTOR_RIGHT_PIN, LOW);
                digitalWrite(STEERING_MOTOR_LEFT_PIN, LOW);
                digitalWrite(DRIVE_MOTOR_ENABLE_PIN, HIGH);
                delay(460);
                motorsOff();
                //delay(2000);
            }
            else{
              //already stopped, motors are off
            }
        }
        
        previousMoveDirection=moveDirection;
    }
}

//Sets both steering and drive motors off
void motorsOff(){
  
    digitalWrite(DRIVE_MOTOR_ENABLE_PIN, LOW);
    digitalWrite(DRIVE_MOTOR_FORWARD_PIN, LOW);
    digitalWrite(DRIVE_MOTOR_REVERSE_PIN, LOW);
    digitalWrite(STEERING_MOTOR_RIGHT_PIN, LOW);
    digitalWrite(STEERING_MOTOR_LEFT_PIN, LOW);
}

//Check if metal detector has found metal
void checkMetal(){
  
 /* int loopsToUse;
  
  if(stopMetal==true){
    //Within timelimit    
    if((stopMetalTime+3000)>millis()){
        loopsToUse=200;
    }
    //Not within timelimit
    else{
      stopMetal=false;
      loopsToUse=10;
    }
  }
  else{//number of averages to use
      loopsToUse=10;
  }*/
  
  
  int threshToUse;
  if((moveDirection==FORWARDLEFT)||(moveDirection==FORWARDRIGHT)||(moveDirection==REVERSERIGHT)||(moveDirection==REVERSELEFT)){
        threshToUse=metalThresh+45;
  }
  else if((moveDirection==FORWARD)||(moveDirection==REVERSE)){
        threshToUse=metalThresh;
  }
  else{
        threshToUse=metalThresh;
  }
  
  
  int flag=0;
  //if(analogRead(METAL_DETECTOR_PIN)>threshToUse){
      int tempVal=0;
      for(int i=0;i<10;i++){
          tempVal=tempVal+analogRead(METAL_DETECTOR_PIN);
          //delay(5);
      }
      if((tempVal/10)>threshToUse){
          flag=1;
      }
  //}
 /* if(previousMoveDirection==-1){
      return;
  }*/
  
    if(flag==1){//(analogRead(METAL_DETECTOR_PIN)>metalThresh){//When connected to metal detector use <
        if(metalFoundAlready==false){
            //New instance of metal was found
            long temp;
            int moveDirection2;
            //Turn on Metal Found LED
            digitalWrite(LED_METAL_FOUND,HIGH);
            
            //Handle time while AMDV is doing metal in automode
            if(mode==AUTOMODE){
                temp = autoModeTimeToNext-millis();
                //Get the current move direction
                moveDirection2=moveDirection;
            } 
            
            //Stop the AMDV
            moveDirection=STOP;            
            setMotors();
        
            //Get GPS Location and Send to computer
            do_GPS();
            
            //Set this metal already found
            metalFoundAlready=true;
            
            //Restart the AMDV
            if(mode==AUTOMODE){
                //Update the time for AMDV operation
                autoModeTimeToNext=millis()+temp;
                //Reset AMDV motion
                moveDirection=moveDirection2;
                setMotors();
            }
        }
    }
    else{
        //Reset metal detection alerts
        if(metalFoundAlready==true){
            //First time not finding metal
            digitalWrite(LED_METAL_FOUND,LOW);
            metalFoundAlready=false;
                     
            int maxFailures = 3;
            Serial.print(METALNOTFOUND,BYTE);
            tempTime=millis()+RESEND_WAIT_TIME;//Wait
            while(true){
                if(Serial.available()>0){
                    if(METALNOTFOUND==Serial.read()){
                        break;
                    }
                }
                if(tempTime<millis()){
                    Serial.print(METALNOTFOUND,BYTE);
                    tempTime=millis()+RESEND_WAIT_TIME;//Wait
                    maxFailures--;
                    if(maxFailures<=0){
                        //Failed to send
                        checkConnection();
                        maxFailures=3;
                    }
                }
            }                
        }       
    }
  
}

void do_GPS(){
    if(GPS_Online==true){
        int maxFailures = 3;
        Serial.print(METALFOUND,BYTE);
        tempTime=millis()+RESEND_WAIT_TIME;//Wait
        while(true){
              if(Serial.available()>0){
                   if(METALFOUND==Serial.read()){
                        break;
                    }
              }
              if(tempTime<millis()){
                    Serial.print(METALFOUND,BYTE);
                    tempTime=millis()+RESEND_WAIT_TIME;//Wait
                    maxFailures--;
                    if(maxFailures<=0){
                        //Failed to send
                        checkConnection();
                        maxFailures=3;
                    }
              }
        }
  
        //Get GPS Location
        get_GPS();
        //GPS is on, so send gps data
        if(GPS_Online==true){
            int maxFailures = 3;
            send_GPS();
            tempTime=millis()+RESEND_WAIT_TIME;//Wait
            while(true){
              if(Serial.available()>0){
                   if(GPSBEGIN==Serial.read()){
                        break;
                    }
              }
              if(tempTime<millis()){
                    send_GPS();
                    tempTime=millis()+RESEND_WAIT_TIME;//Wait
                    maxFailures--;
                    if(maxFailures<=0){
                        //Failed to send
                        checkConnection();
                        maxFailures=3;
                    }
              }
            }
        }
        //GPS is not on, so alert user to that
        else{
            Serial.print("$$GPS Data Not Available%");
            int maxFailures = 3;
            int data;
            tempTime=millis()+RESEND_WAIT_TIME;//Wait
            while(true){
              if(Serial.available()>0){
                    data = Serial.read();
                    if(METALNOGPS==data){
                        //Continue
                        break;
                    }
                    else if(GPSBEGIN==data){
                        continue;
                    }
                    else if(AMDVPING==data){
                        maxFailures=3;
                        tempTime=millis()+RESEND_WAIT_TIME;
                    }
              }
              if(tempTime<millis()){
                    Serial.print(AMDVPING,BYTE);
                    maxFailures=maxFailures-1;
                    if(maxFailures<=0){
                        reset_AMDV(false);
                    }
                    tempTime = millis()+RESEND_WAIT_TIME;
              }
            }
            
            
          }
    }
    else{
        //Send Metal Found Alert
        int maxFailures = 3;
        int flag=0;
        int data;
        Serial.print(METALFOUND,BYTE);
        tempTime=millis()+RESEND_WAIT_TIME;//Wait
        while(true){
              if(Serial.available()>0){
                   data=Serial.read();
                   if(METALFOUND==data){
                       flag=1;
                    }
                    else if(METALNOGPS==data){
                       break;//Continue operation
                    }
                    else if(AMDVPING==data){
                        maxFailures=3;
                        tempTime=millis()+RESEND_WAIT_TIME;
                    }
              }
              if(tempTime<millis() && flag==0){
                    Serial.print(METALFOUND,BYTE);
                    tempTime=millis()+RESEND_WAIT_TIME;//Wait
                    maxFailures--;
                    if(maxFailures<=0){
                        //Failed to send
                        checkConnection();
                        maxFailures=3;
                    }
              }
              else if(tempTime<millis() && flag==1){
                    Serial.print(AMDVPING,BYTE);
                    maxFailures=maxFailures-1;
                    if(maxFailures<=0){
                        reset_AMDV(false);
                    }
                    tempTime = millis()+RESEND_WAIT_TIME;
              }
        }
    }   
}

//Send the GPS location
void send_GPS(){
            //Send GPS Location
            Serial.print(GPSBEGIN);
            Serial.print(lat);
            Serial.print(GPSLAT);
            Serial.print(lon);
            Serial.print(GPSLONG); 
}

//Gets the most upto date GPS daa
void get_GPS(){
    unsigned long toExitLoopTime;
    boolean tooMuchTimeElapsed=false;
    int flag=0;
  
    /**********************************************************************************
        Possibly clear the RX Buffer of GPS?
    ***********************************************************************************/ 
  
    toExitLoopTime=millis()+GPS_WAIT_TIME;//time to wait for valid fix
  
    while(tooMuchTimeElapsed==false){
        while(gpsSerial.available()){
            flag =1;
            //Get GPS data and process it
            if(gps.encode(gpsSerial.read())==true){  
                gps.get_position(&lat,&lon,&fix_age);//Get position and fix age
                if(fix_age<500){
                    //Have recieved location from satellites
                    tooMuchTimeElapsed=true;
                    break;
                }
            }
            else{//iffix_age>500 || fix_age==TinyGPS::GPS_INVALID_AGE)
                    //No current fix
                    if(toExitLoopTime<millis()){
                         GPS_Online=false;
                         digitalWrite(LED_GPS_ON,LOW);
                         tooMuchTimeElapsed=true;
                         break;
                    } 
            }     
        }
        //GPS has managed to become disconnected
        if(flag == 0 && toExitLoopTime<millis()){
            GPS_Online=false;
            digitalWrite(LED_GPS_ON,LOW);
            tooMuchTimeElapsed=true;
        }
    }
}

int main(void)
{
	init();

	setup();
    
	for (;;)
		loop();
        
	return 0;
}

