package com.team5.aLife.BaseStation;

import android.content.ContentValues;
import android.content.Context;
import android.content.Intent;
import android.database.Cursor;
import android.database.SQLException;
import android.database.sqlite.SQLiteDatabase;
import android.database.sqlite.SQLiteOpenHelper;
import android.provider.BaseColumns;
import android.util.Log;

public class DBHandler 
{   
	/*
	 * No foreign key's were added to the database schema because sqlite would create a restraint
	 * where you could not delete a row form a table where a dependency existed.  So if there were
	 * entry's in our notification history that included userId's and we wanted to delete a user
	 * from our user table, then we would have to systematically find all instances of that user
	 * in our notification history table and delete those prior to deleting the user from our 
	 * user table.  We did not feel this would be the best model to follow.
	 *
	 */
	
	private Intent myIntent;
	
    private static final String TAG = "DatabaseHelper";
    
    protected static final String DATABASE_NAME = "aLife.db";
    protected static final String DATABASE_TABLE_USERS = "users";
    protected static final String DATABASE_TABLE_DEVICES = "devices";
    protected static final String DATABASE_TABLE_POWER_HISTORY = "power_history";
    protected static final String DATABASE_TABLE_NOTIFICATION_HISTORY = "notification_history";
    protected static final String DATABASE_TABLE_NOTIFICATION_SETUP = "notification_setup";
    
    private static final int DATABASE_VERSION = 1;
    
    //  SQL string to create the Users Table
    private static final String CREATE_TABLE_USERS = 
    	"CREATE TABLE " + DATABASE_TABLE_USERS + " ("
	    + DB_Users._ID + " INTEGER PRIMARY KEY,"
	    + DB_Users.USER_NAME + " TEXT UNIQUE,"
	    + DB_Users.PASSWORD + " TEXT,"
	    + DB_Users.PERMISSION + " TEXT,"
	    + DB_Users.FAILED_LOGIN_NUM + " INTEGER,"
	    + DB_Users.TIMEOUT + " INTEGER,"
	    + DB_Users.TIMEOUT_TIME + " INTEGER"
	    + ");";

    //  SQL string to create the Devices Table
    private static final String CREATE_TABLE_DEVICES = 
    	"CREATE TABLE " + DATABASE_TABLE_DEVICES + " ("
	    + DB_Devices._ID + " INTEGER PRIMARY KEY,"
	    + DB_Devices.DEVICE_TYPE + " INTEGER,"
	    + DB_Devices.NAME + " TEXT,"
	    + DB_Devices.ADDRESS + " TEXT,"
	    + DB_Devices.ZIGBEE_STATUS + " INTEGER,"
	    + DB_Devices.STATUS + " INTEGER,"
	    + DB_Devices.UPDATED_TIME + " INTEGER"
	    + ");";
    
    //  SQL string to create the Notifications Setup Table
    private static final String CREATE_TABLE_NOTIFICATIONS_SETUP = 
    	"CREATE TABLE " + DATABASE_TABLE_NOTIFICATION_SETUP + " ("
	    + DB_Notifications_Setup._ID + " INTEGER PRIMARY KEY,"
	    + DB_Notifications_Setup.DEVICE_ID + " INTEGER,"
	    + DB_Notifications_Setup.USER_ID + " INTEGER,"
	    + DB_Notifications_Setup.NOTIFICATION_TYPE + " INTEGER,"
	    + DB_Notifications_Setup.TRIGGERING_CONDITION + " TEXT"
	    + ");";
    
    //  SQL string to create the Notifications History Table
    private static final String CREATE_TABLE_NOTIFICATIONS_HISTORY = 
    	"CREATE TABLE " + DATABASE_TABLE_NOTIFICATION_HISTORY + " ("
	    + DB_Notifications_History._ID + " INTEGER PRIMARY KEY,"
	    + DB_Notifications_History.NOTIFICATION_ID + " INTEGER,"
	    + DB_Notifications_History.SATISFIED + " INTEGER,"
	    + DB_Notifications_History.TIME + " INTEGER"
	    + ");";
    
    //  SQL string to create the Power History Table
    private static final String CREATE_TABLE_POWER_HISTORY = 
    	"CREATE TABLE " + DATABASE_TABLE_POWER_HISTORY + " ("
	    + DB_Power_History._ID + " INTEGER PRIMARY KEY,"
	    + DB_Power_History.TIME + " INTEGER,"
	    + DB_Power_History.DEVICE_ID + " INTEGER,"
	    + DB_Power_History.POWER_CONSUMPTION + " INTEGER"
	    + ");";
        
    private final Context context; 
    
    private DatabaseHelper DBHelper;
    private SQLiteDatabase db;

    public DBHandler(Context ctx) 
    {
        this.context = ctx;
        DBHelper = new DatabaseHelper(context);
    }
        
    /**
     * This class helps open, create, and upgrade the database file.
     */
    private static class DatabaseHelper extends SQLiteOpenHelper 
    {
        DatabaseHelper(Context context) 
        {
            super(context, DATABASE_NAME, null, DATABASE_VERSION);
        }

        @Override
        public void onCreate(SQLiteDatabase db) 
        {
        	Log.i("DatabaseHelper:  onCreate", "I about to create the table");
            db.execSQL(CREATE_TABLE_USERS);
            db.execSQL(CREATE_TABLE_DEVICES);
            db.execSQL(CREATE_TABLE_NOTIFICATIONS_SETUP);
            db.execSQL(CREATE_TABLE_NOTIFICATIONS_HISTORY);
            db.execSQL(CREATE_TABLE_POWER_HISTORY);
        }

        @Override
        public void onUpgrade(SQLiteDatabase db, int oldVersion, 
        int newVersion) 
        {
            Log.w(TAG, "Upgrading database from version " + oldVersion 
                    + " to "
                    + newVersion + ", which will destroy all old data");
            db.execSQL("DROP TABLE IF EXISTS " + DATABASE_NAME);
            onCreate(db);
        }
    }    
    
    //---opens the database---
    public DBHandler open() throws SQLException 
    {
        db = DBHelper.getWritableDatabase();
        return this;
    }

    //---closes the database---    
    public void close() 
    {
        DBHelper.close();
    }
    
    ///----------------------------Methods for Devices Table------------------------------------
    
    //---Insert a device into the table---
    public long insertDevice(Device device) 
    {
        ContentValues initialValues = new ContentValues();
        initialValues.put(DB_Devices.DEVICE_TYPE, device.getDeviceType());
        initialValues.put(DB_Devices.NAME, device.getDeviceName());
        initialValues.put(DB_Devices.ADDRESS, device.getDeviceAddress());
        initialValues.put(DB_Devices.ZIGBEE_STATUS, device.getZigbeeStatus());
        initialValues.put(DB_Devices.STATUS, device.getDeviceStatus());
        initialValues.put(DB_Devices.UPDATED_TIME, device.getUpdatedTime());
        return db.insert(DATABASE_TABLE_DEVICES, null, initialValues);
    }

    //---deletes a particular title---
    public boolean deleteDevice(long rowId) 
    {
        return db.delete(DATABASE_TABLE_DEVICES, DB_Devices._ID + 
        		"=" + rowId, null) > 0;
    }

    //---retrieves all the registered devices---
    public Cursor getAllDevices() 
    {
        return db.query(DATABASE_TABLE_DEVICES, new String[] {
        		DB_Devices._ID, 
        		DB_Devices.DEVICE_TYPE,
        		DB_Devices.NAME,
        		DB_Devices.ADDRESS,
        		DB_Devices.ZIGBEE_STATUS,
        		DB_Devices.STATUS,
        		DB_Devices.UPDATED_TIME}, 
                null, 
                null, 
                null, 
                null, 
                null);
    }

    //---retrieves a particular device---
    public Cursor getDevice(long rowId) throws SQLException 
    {
        Cursor mCursor =
                db.query(true, DATABASE_TABLE_DEVICES, new String[] {
                		DB_Devices._ID, 
                		DB_Devices.DEVICE_TYPE,
                		DB_Devices.NAME,
                		DB_Devices.ADDRESS,
                		DB_Devices.ZIGBEE_STATUS,
                		DB_Devices.STATUS,
                		DB_Devices.UPDATED_TIME},
                		DB_Devices._ID + "=" + rowId, 
                		null,
                		null, 
                		null, 
                		null, 
                		null);
        return mCursor;
    }

    //---updates a device---
    public boolean updateDevice(Device device) 
    {
        ContentValues args = new ContentValues();
        args.put(DB_Devices.DEVICE_TYPE, device.getDeviceType());
        args.put(DB_Devices.NAME, device.getDeviceName());
        args.put(DB_Devices.ADDRESS, device.getDeviceAddress());
        args.put(DB_Devices.ZIGBEE_STATUS, device.getZigbeeStatus());
        args.put(DB_Devices.STATUS, device.getDeviceStatus());
        args.put(DB_Devices.UPDATED_TIME, device.getUpdatedTime());
        return db.update(DATABASE_TABLE_DEVICES, args, 
        				 DB_Devices._ID + "=" + device.getDeviceID(), null) > 0;
    }
    
  ///----------------------------------------------------------------------------------------------
    
  ///----------------------------Methods for Users Table------------------------------------
    
    //---Insert a user into the table---
    public long insertUser(String userName, String password, String permission, 
    					   int failedLoginNum, int timeout, int timeoutTime) 
    {
        ContentValues initialValues = new ContentValues();
        initialValues.put(DB_Users.USER_NAME, userName);
        initialValues.put(DB_Users.PASSWORD, password);
        initialValues.put(DB_Users.PERMISSION, permission);
        initialValues.put(DB_Users.FAILED_LOGIN_NUM, failedLoginNum);
        initialValues.put(DB_Users.TIMEOUT, timeout);
        initialValues.put(DB_Users.TIMEOUT_TIME, timeoutTime);
        return db.insert(DATABASE_TABLE_USERS, null, initialValues);
    }

    //---deletes a particular user---
    public boolean deleteUser(String userName) 
    {
        return db.delete(DATABASE_TABLE_USERS, DB_Users.USER_NAME + 
        		"=" + "'"+userName+"'", null) > 0;
    }

    //---retrieves a particular user---
    public Cursor getAllUsers() throws SQLException 
    {	
       return db.query(DATABASE_TABLE_USERS, new String[] {
    		   		   DB_Users._ID, 
    		   		   DB_Users.USER_NAME,
    		   		   DB_Users.PASSWORD,
    		   		   DB_Users.PERMISSION,
    		   		   DB_Users.FAILED_LOGIN_NUM,
    		   		   DB_Users.TIMEOUT,
    		   		   DB_Users.TIMEOUT_TIME}, 
			           null, 
			           null, 
			           null, 
			           null, 
			           null);
    }
    
    //---retrieves all users in the database---
    public Cursor getUser(String userName) throws SQLException 
    {
        Cursor mCursor =
                db.query(true, DATABASE_TABLE_USERS, new String[] {
                		DB_Users._ID, 
                		DB_Users.USER_NAME,
                		DB_Users.PASSWORD,
                		DB_Users.PERMISSION,
                		DB_Users.FAILED_LOGIN_NUM,
                		DB_Users.TIMEOUT,
                		DB_Users.TIMEOUT_TIME},
                		DB_Users.USER_NAME + "=" + "'"+userName+"'", 
                		null,
                		null, 
                		null, 
                		null, 
                		null);
        return mCursor;
    }
    
    
    //---updates a user---
    public boolean updateUser(long rowId, String userName, String password, String permission, 
			   					int failedLoginNum, int timeout, int timeoutTime) 
    {
        ContentValues args = new ContentValues();
        args.put(DB_Users.USER_NAME, userName);
        args.put(DB_Users.PASSWORD, password);
        args.put(DB_Users.PERMISSION, permission);
        args.put(DB_Users.FAILED_LOGIN_NUM, failedLoginNum);
        args.put(DB_Users.TIMEOUT, timeout);
        args.put(DB_Users.TIMEOUT_TIME, timeoutTime);
        return db.update(DATABASE_TABLE_USERS, args, 
        				 DB_Users._ID + "=" + rowId, null) > 0;
    }
    
  ///----------------------------------------------------------------------------------------------   
  
  ///----------------------------Methods for Notification Setup Table------------------------------
    
    //---Setup up a notification and store it into the table---
    public long insertNotification(ALifeNotificationSetup notification) 
    {
        ContentValues initialValues = new ContentValues();
        initialValues.put(DB_Notifications_Setup.DEVICE_ID, notification.getDeviceID());
        initialValues.put(DB_Notifications_Setup.USER_ID, notification.getUserID());
        initialValues.put(DB_Notifications_Setup.NOTIFICATION_TYPE, notification.getNotificationType());
        initialValues.put(DB_Notifications_Setup.TRIGGERING_CONDITION, notification.getTriggeringCondition());
        return db.insert(DATABASE_TABLE_NOTIFICATION_SETUP, null, initialValues);
    }

    //---deletes a particular notification---
    public boolean deleteNotification(long rowId) 
    {
        return db.delete(DATABASE_TABLE_NOTIFICATION_SETUP, DB_Notifications_Setup._ID + 
        		"=" + rowId, null) > 0;
    }
    
    //---deletes a particular notification---
    public boolean deleteNotificationByDeviceID(long deviceID) 
    {
        return db.delete(DATABASE_TABLE_NOTIFICATION_SETUP, DB_Notifications_Setup.DEVICE_ID + 
        		"=" + deviceID, null) > 0;
    }

    //---retrieves a particular notification---
    public Cursor getNotification(long rowId) throws SQLException 
    {
        Cursor mCursor =
                db.query(true, DATABASE_TABLE_NOTIFICATION_SETUP, new String[] {
                		DB_Notifications_Setup._ID, 
                		DB_Notifications_Setup.DEVICE_ID,
                		DB_Notifications_Setup.USER_ID,
                		DB_Notifications_Setup.NOTIFICATION_TYPE,
                		DB_Notifications_Setup.TRIGGERING_CONDITION},
                		DB_Notifications_Setup._ID + "=" + rowId, 
                		null,
                		null, 
                		null, 
                		null, 
                		null);
        return mCursor;
    }
    
    //---retrieves a particular notification---
    public Cursor getNotification(int deviceID, int userID) throws SQLException 
    {
        Cursor mCursor =
                db.query(true, DATABASE_TABLE_NOTIFICATION_SETUP, new String[] {
                		DB_Notifications_Setup._ID, 
                		DB_Notifications_Setup.DEVICE_ID,
                		DB_Notifications_Setup.USER_ID,
                		DB_Notifications_Setup.NOTIFICATION_TYPE,
                		DB_Notifications_Setup.TRIGGERING_CONDITION},
                		DB_Notifications_Setup.DEVICE_ID + "=" + deviceID + " AND " +
                		DB_Notifications_Setup.USER_ID + "=" + userID, 
                		null,
                		null, 
                		null, 
                		null, 
                		null);
        return mCursor;
    }
    
    //---retrieves all the set up notifications---
    public Cursor getAllNotifications() 
    {
        return db.query(DATABASE_TABLE_NOTIFICATION_SETUP, new String[] {
        		DB_Notifications_Setup._ID, 
        		DB_Notifications_Setup.DEVICE_ID,
        		DB_Notifications_Setup.USER_ID,
        		DB_Notifications_Setup.NOTIFICATION_TYPE,
        		DB_Notifications_Setup.TRIGGERING_CONDITION},
                null, 
                null, 
                null, 
                null, 
                null);
    }
    
    //---retrieves all the set up notifications for a user---
    public Cursor getAllNotificationsForUser(int userID)
    {
    	Cursor mCursor =
            db.query(true, DATABASE_TABLE_NOTIFICATION_SETUP, new String[] {
            		DB_Notifications_Setup._ID, 
            		DB_Notifications_Setup.DEVICE_ID,
            		DB_Notifications_Setup.USER_ID,
            		DB_Notifications_Setup.NOTIFICATION_TYPE,
            		DB_Notifications_Setup.TRIGGERING_CONDITION},
            		DB_Notifications_Setup.USER_ID + "=" + userID, 
            		null,
            		null, 
            		null, 
            		null, 
            		null);
    	return mCursor;
    }

    //---updates a notification---
    public boolean updateNotification(ALifeNotificationSetup notification) 
    {
        ContentValues args = new ContentValues();
        args.put(DB_Notifications_Setup.DEVICE_ID, notification.getDeviceID());
        args.put(DB_Notifications_Setup.USER_ID, notification.getUserID());
        args.put(DB_Notifications_Setup.NOTIFICATION_TYPE, notification.getNotificationType());
        args.put(DB_Notifications_Setup.TRIGGERING_CONDITION, notification.getTriggeringCondition());
        return db.update(DATABASE_TABLE_NOTIFICATION_SETUP, args, 
        				 DB_Notifications_Setup._ID + "=" + notification.getID(), null) > 0;
    }
    
  ///---------------------------------------------------------------------------------------------- 
 
  ///----------------------------Methods for Notification History Table----------------------------
    
    //---Insert sent notification into the log---
    public long insertNotificationLogEntry(int notificationID, int satisfied, int time) 
    {
        ContentValues initialValues = new ContentValues();
        initialValues.put(DB_Notifications_History.NOTIFICATION_ID, notificationID);
        initialValues.put(DB_Notifications_History.SATISFIED, satisfied);
        initialValues.put(DB_Notifications_History.TIME, time);
        return db.insert(DATABASE_TABLE_NOTIFICATION_HISTORY, null, initialValues);
    }

    //---Deletes a particular notification log entry---
    public boolean deleteNotificationLogEntry(long rowId) 
    {
        return db.delete(DATABASE_TABLE_NOTIFICATION_HISTORY, DB_Notifications_History._ID + 
        		"=" + rowId, null) > 0;
    }
    
    //---Deletes entries for logs that are not within the lastXDays
    public boolean deleteNotificationLogEntriesWithin(int lastXDays) 
    {
        return db.delete(DATABASE_TABLE_NOTIFICATION_HISTORY, DB_Notifications_History.TIME + 
        		"<" + lastXDays, null) > 0;
    }

    //---Retrieves a particular notification log entry---
    public Cursor getNotificationLogEntry(long rowId) throws SQLException 
    {
        Cursor mCursor =
                db.query(true, DATABASE_TABLE_NOTIFICATION_HISTORY, new String[] {
                		DB_Notifications_History._ID, 
                		DB_Notifications_History.NOTIFICATION_ID,
                		DB_Notifications_History.SATISFIED,
                		DB_Notifications_History.TIME},
                		DB_Notifications_History._ID + "=" + rowId, 
                		null,
                		null, 
                		null, 
                		null, 
                		null);
        return mCursor;
    }
    
    //---Retrieves all the notifications in the log---
    public Cursor getNotificationLog() 
    {
        return db.query(DATABASE_TABLE_NOTIFICATION_HISTORY, new String[] {
        		DB_Notifications_Setup._ID,
        		DB_Notifications_History.NOTIFICATION_ID,
        		DB_Notifications_History.SATISFIED,
        		DB_Notifications_History.TIME},
                null, 
                null, 
                null, 
                null, 
                null);
    }
    

    //---updates a notification log---
    public boolean updateNotificationLogEntry(int rowID, int notificationID, int satisfied, int time) 
    {
        ContentValues args = new ContentValues();
        args.put(DB_Notifications_History.NOTIFICATION_ID, notificationID);
        args.put(DB_Notifications_History.SATISFIED, satisfied);
        args.put(DB_Notifications_History.TIME, time);
        return db.update(DATABASE_TABLE_NOTIFICATION_HISTORY, args, 
        				 DB_Notifications_History._ID + "=" + rowID, null) > 0;
    }
    
  ///---------------------------------------------------------------------------------------------- 
    
    
///----------------------------Methods for Power History Table-------------------------------------
    
    //---Insert power history information into the log---
    public long insertPowerHistoryLogEntry(int deviceID, int time, int powerConsumption) 
    {
        ContentValues initialValues = new ContentValues();
        initialValues.put(DB_Power_History.TIME, time);
        initialValues.put(DB_Power_History.DEVICE_ID, deviceID);
        initialValues.put(DB_Power_History.POWER_CONSUMPTION, powerConsumption);
        return db.insert(DATABASE_TABLE_POWER_HISTORY, null, initialValues);
    }

    //---Deletes a particular power history log entry---
    public boolean deletePowerHistoryLogEntry(long rowId) 
    {
        return db.delete(DATABASE_TABLE_POWER_HISTORY, DB_Power_History._ID + 
        		"=" + rowId, null) > 0;
    }
    
    //---Deletes entries for logs that are not within the lastXDays
    public boolean deletePowerHistoryLogEntriesWithin(int lastXDays) 
    {
        return db.delete(DATABASE_TABLE_POWER_HISTORY, DB_Power_History.TIME + 
        		"<" + lastXDays, null) > 0;
    }

    //---Retrieves a particular power history log entry---
    public Cursor getPowerHistoryLogEntry(long rowId) throws SQLException 
    {
        Cursor mCursor =
                db.query(true, DATABASE_TABLE_POWER_HISTORY, new String[] {
                		DB_Power_History._ID, 
                		DB_Power_History.TIME,
                		DB_Power_History.DEVICE_ID,
                		DB_Power_History.POWER_CONSUMPTION},
                		DB_Power_History._ID + "=" + rowId, 
                		null,
                		null, 
                		null, 
                		null, 
                		null);
        return mCursor;
    }
    
    //---Retrieves all the power entries in the log---
    public Cursor getPowerHistoryLog() 
    {
        return db.query(DATABASE_TABLE_POWER_HISTORY, new String[] {
        		DB_Power_History._ID, 
        		DB_Power_History.TIME,
        		DB_Power_History.DEVICE_ID,
        		DB_Power_History.POWER_CONSUMPTION},
                null, 
                null, 
                null, 
                null, 
                null);
    }
    

    //---updates a notification log---
    public boolean updatePowerHistoryLogEntry(int rowID, int deviceID, int time, int powerConsumption) 
    {
        ContentValues args = new ContentValues();
        args.put(DB_Power_History.TIME, time);
        args.put(DB_Power_History.DEVICE_ID, deviceID);
        args.put(DB_Power_History.POWER_CONSUMPTION, powerConsumption);
        return db.update(DATABASE_TABLE_POWER_HISTORY, args, 
        				 DB_Power_History._ID + "=" + rowID, null) > 0;
    }
    
  ///----------------------------------------------------------------------------------------------   
    /**
     * User table
     */
    public static final class DB_Users implements BaseColumns {
        // This class cannot be instantiated
        private DB_Users() {}

        /**
         * The user name
         * <P>Type: TEXT</P>
         */
        public static final String USER_NAME = "user_name";

        /**
         * The password
         * <P>Type: TEXT</P>
         */
        public static final String PASSWORD = "password";

        /**
         * The permission level of the user
         * <P>Type: TEXT</P>
         */
        public static final String PERMISSION = "permission";

        /**
         * The amount of failed login attempts
         * <P>Type: INTEGER </P>
         */
        public static final String FAILED_LOGIN_NUM = "failed_login_num";
        
        /**
         * A boolean to tell whether a timeout has been activated
         * <P>Type: INTEGER </P>
         */
        public static final String TIMEOUT = "timeout";
        
        /**
         * The time when the timeout period started
         * <P>Type: INTEGER </P>
         */
        public static final String TIMEOUT_TIME = "timeout_time";
    }
    
    /**
     * Devices table
     */
    public static final class DB_Devices implements BaseColumns {
        // This class cannot be instantiated
        private DB_Devices() {}

        /**
         * The device type
         * <P>Type: TEXT</P>
         */
        public static final String DEVICE_TYPE = "device_type";

        /**
         * The name the user provides to the device
         * <P>Type: TEXT</P>
         */
        public static final String NAME = "name";

        /**
         * The zigbee address of the device
         * <P>Type: TEXT</P>
         */
        public static final String ADDRESS = "address";

        /**
         * The zigbee status of the device
         * <P>Type: INTEGER </P>
         */
        public static final String ZIGBEE_STATUS = "zigbee_status";
        
        /**
         * The status of the device (ex: on/off)
         * <P>Type: INTEGER </P>
         */
        public static final String STATUS = "status";
        
        /**
         * The time in which the entry was made
         * <P>Type: INTEGER </P>
         */
        public static final String UPDATED_TIME = "updated_time";
        
    }
    
    /**
     * Notification table
     */
    public static final class DB_Notifications_Setup implements BaseColumns {
        // This class cannot be instantiated
        private DB_Notifications_Setup() {}

        /**
         * The ID of the device the notification is referencing
         * <P>Type: INTEGER</P>
         */
        public static final String DEVICE_ID = "device_id";

        /**
         * The IDs of all users registered for this notification
         * <P>Type: INTEGER</P>
         */
        public static final String USER_ID = "user_id";

        /**
         * The notification type
         * <P>Type: INTEGER</P>
         */
        public static final String NOTIFICATION_TYPE = "notification_type";

        /**
         * The condition which will trigger the notification
         * <P>Type: TEXT </P>
         */
        public static final String TRIGGERING_CONDITION = "triggering_condition";
        
    }
    
    /**
     * Notification History table
     */
    public static final class DB_Notifications_History implements BaseColumns {
        // This class cannot be instantiated
        private DB_Notifications_History() {}

        /**
         * The ID of the particular notification that was logged
         * <P>Type: INTEGER</P>
         */
        public static final String NOTIFICATION_ID = "notification_id";
        
        /**
         * The tells whether or not the sent notification has been responded to
         * <P>Type: INTEGER</P>
         */
        public static final String SATISFIED = "satisfied";
        
        /**
         * The time in which the notification entry was logged
         * <P>Type: INTEGER</P>
         */
        public static final String TIME = "time";
        
    }
    
    /**
     * Power History table
     */
    public static final class DB_Power_History implements BaseColumns {
        // This class cannot be instantiated
        private DB_Power_History() {}
        
        /**
         * The ID of the device that was polled
         * <P>Type: INTEGER</P>
         */
        public static final String DEVICE_ID = "device_id";
        
        /**
         * The time in which the power entry was polled
         * <P>Type: INTEGER</P>
         */
        public static final String TIME = "time";
        
        /**
         * The amount of power the device has used
         * <P>Type: INTEGER</P>
         */
        public static final String POWER_CONSUMPTION = "power_consumption";
        
    }
}
