#include "mainwindow.h"
#include "ui_mainwindow.h"

#include <QtMultimediaWidgets/QVideoWidget>
#include <QtMultimedia/QMediaPlayer>
#include <QtMultimedia/QMediaPlaylist>
#include <QFile>
#include <QHBoxLayout>
#include <QPainter>
#include "controllerthread.h"
#include <udpclient.h>
#include <QSettings>
#include <unistd.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

MainWindow::MainWindow(QWidget *parent) :
    QMainWindow(parent),
    ui(new Ui::MainWindow)
{
    ui->setupUi(this);

    m_sSettingsFile = QApplication::applicationDirPath() + "/config.ini";
    if(access(m_sSettingsFile.toStdString().c_str(), F_OK ) == -1){
        qWarning("Failed to access config file");
        this->close();
    }else
        qDebug("Config found");

    QWidget *widget = ui->VideoWidget;
    if(widget == nullptr){
        qFatal("widget is null");
    }
    QSettings settings(m_sSettingsFile, QSettings::NativeFormat);

    ip_addr = settings.value("ip", "").toString();
    if(ip_addr.isEmpty()){
        qWarning("Could not load ip address from config, using localhost");
        ip_addr = "127.0.0.1";
    }else{
        if(inet_addr(ip_addr.toStdString().c_str()) == INADDR_NONE)
            qFatal("Invalid ip address %s", ip_addr.toStdString().c_str());
        else
            qDebug() << "Using remote ip address" << ip_addr;
    }

    client = new UDPClient(ip_addr);
    player = new QMediaPlayer;
    vw = new QVideoWidget;
    layout = new QHBoxLayout;

    layout->addWidget(vw);
    widget->setLayout(layout);
    player->setVideoOutput(vw);

    stream_url = settings.value("url", "").toString();
    player->setMedia(QUrl(stream_url));
    player->play();

    ui->LabelControllerStatus->setStyleSheet("QLabel { background-color: rgb(209, 101, 101)}");
    ui->TankBotStates->addItem("Standby");
    ui->TankBotStates->addItem("Follow");
    ui->TankBotStates->addItem("Attack");
}

MainWindow::~MainWindow()
{
    delete ui;
}

void MainWindow::ControllerConnected(bool connected)
{
    int index = -1;
    if(connected){
        ui->LabelControllerStatus->setText("Connected");
        ui->LabelControllerStatus->resize(71,18);
        ui->LabelControllerStatus->setStyleSheet("QLabel { background-color: rgb(146, 191, 64)}");
        ui->TankBotStates->addItem("Remote Control");
        ui->TankBotStates->setCurrentIndex(ui->TankBotStates->findText("Remote Control"));
        ui->ButtonFire->setEnabled(false);
        ui->TankBotStates->setEnabled(false);
    }else{
        ui->LabelControllerStatus->setText("Disconnected");
        ui->LabelControllerStatus->resize(87,18);
        ui->LabelControllerStatus->setStyleSheet("QLabel { background-color: rgb(209, 101, 101)}");
        ui->TankBotStates->setCurrentIndex(ui->TankBotStates->findText("Standby"));
        if((index = ui->TankBotStates->findText("Remote Control")) != -1)
            ui->TankBotStates->removeItem(index);
        ui->TankBotStates->setEnabled(true);
    }
}

void MainWindow::ControllerButtonPressed(int button, int state)
{
    switch(button){
    case BUTTON_LB:
        if(state == BUTTON_PRESSED)
            ui->ButtonFire->animateClick();
        break;
    default:
        break;
    }
}

void MainWindow::ControllerUpdateUI(struct TankBotCommand *cmd)
{
    cmd->state = ui->TankBotStates->currentIndex();
    ui->LeftXLabel->setText(QString::number(cmd->moveDir.X));
    ui->LeftYLabel->setText(QString::number(cmd->moveDir.Y));

    ui->RightXLabel->setText(QString::number(cmd->lookDir.X));
    ui->RightYLabel->setText(QString::number(cmd->lookDir.Y));
    client->sendBytes(cmd, sizeof(struct TankBotCommand));
}

void MainWindow::on_ButtonRefresh_clicked()
{
    QSettings settings(m_sSettingsFile, QSettings::NativeFormat);
    stream_url = settings.value("url", "").toString();
    player->stop();
    player->setMedia(QUrl(stream_url));
    player->play();
}

void MainWindow::on_TankBotStates_currentIndexChanged(int index)
{
    struct TankBotCommand cmd;
    memset (&cmd, 0, sizeof(struct TankBotCommand));
    if(index == TANKBOT_ATTACK)
        ui->ButtonFire->setEnabled(true);
    else
        ui->ButtonFire->setEnabled(false);
    cmd.state = index;
    client->sendBytes(&cmd, sizeof(struct TankBotCommand));
}

void MainWindow::on_ButtonFire_pressed()
{
    struct TankBotCommand cmd;
    memset (&cmd, 0, sizeof(struct TankBotCommand));
    cmd.state = ui->TankBotStates->currentIndex();
    cmd.fireState = 1;
    client->sendBytes(&cmd, sizeof(struct TankBotCommand));
}
void MainWindow::on_ButtonFire_released()
{
    struct TankBotCommand cmd;
    memset (&cmd, 0, sizeof(struct TankBotCommand));
    cmd.state = ui->TankBotStates->currentIndex();
    cmd.fireState = 0;
    client->sendBytes(&cmd, sizeof(struct TankBotCommand));
}
