
from django.views.generic.base import TemplateView
from django.contrib import messages
from django.contrib.auth import (get_user_model,
                                authenticate, 
                                login,
                                update_session_auth_hash)
from django.contrib.auth.models import User
from django.contrib.auth.forms import (UserCreationForm, 
                                        AuthenticationForm,
                                        PasswordChangeForm)
from django.shortcuts import render, redirect
from .forms import UserRegisterForm, ProfileEditForm, AddDependentForm
from .models import Dependents
from friendship.models import Friend, Follow, Block, FriendshipRequest

User = get_user_model()

#Login
def login_view(request):
    if request.method == 'POST':
        form = AuthenticationForm(data=request.POST)
        if form.is_valid():
            user = form.get_user()
            login(request, user)
            return redirect('home')
    else:
        form = AuthenticationForm()
    return render(request, 'registration/login.html', { 'form': form })


# Registration

def register(request):
    if request.method == 'POST':
        form = UserRegisterForm(request.POST)
        if form.is_valid():
            form.save()
            messages.success(request, f'Account Created Succesfully!')
            username = request.POST['username']
            password = request.POST['password1']
            user = authenticate(request, username=username, password=password)
            if user is not None:
                login(request, user)
            return redirect('home')
    else:
        form = UserRegisterForm()
    
    context = {'form' : form}
    return render(request, 'accounts/register.html', context)

# View Profile

def view_profile(request):
    context = {'user': request.user}
    return render(request, 'accounts/view_profile.html', context)

# Custom Edit Profile

def edit_profile(request):
    if request.method == 'POST':
        form = ProfileEditForm(request.POST, instance = request.user)

        if form.is_valid():
            form.save()
            messages.success(request, f'Account Updated Succesfully!')
            return redirect('view_profile')

    else:
        form = ProfileEditForm(instance = request.user)
        context = { 'form' : form}
        return render(request, 'accounts/edit_profile.html', context)

# Dependents

def manage_dependents(request):
    all_dependents = Friend.objects.friends(request.user)
    requests = Friend.objects.unread_requests(user=request.user)
    list = Dependents.objects.filter(person=request.user.id)
    if list:
        is_dependent = True
    else:
        is_dependent = False

    context = {
        'is_dependent': is_dependent,
        'all_dependents': all_dependents,
        'requests': requests,
    }

    return render(request, 'accounts/manage_dependents.html', context)

def add_dependent(request):
    if request.method == 'POST':
        form = AddDependentForm(request.POST)
        if form.is_valid():
            username = form.cleaned_data['username']
            try:
                dependent_form = User.objects.get(username=username)
                new_dependent = Dependents(request.user.id, request.POST)
                new_dependent.person = dependent_form
                new_dependent.dependent = True
                new_dependent.save()
                Friend.objects.add_friend(request.user, dependent_form, message='Please add me to your household.')
                messages.success(request, f'Request for dependent sent.')
            except:
                messages.error(request, f'An user with this name does not exist.')
                redirect('add_dependent')
            
            return redirect('home')
    else:
        form = AddDependentForm()
    
    return render(request, 'accounts/add_dependent.html', {'form': form})

def accept_dependent(request):
    friend_request = FriendshipRequest.objects.get(to_user=request.user)
    friend_request.accept()
    messages.success(request, f'You are now a dependent.')
    
    return redirect('home')

def decline_dependent(request):
    friend_request = FriendshipRequest.objects.get(to_user=request.user)
    friend_request.cancel()
    messages.success(request, f'Request for dependent declined.')
    
    return redirect('home')

def delete_dependent(request, pk):
    dependent = User.objects.get(pk=pk)
    Friend.objects.remove_friend(request.user, dependent)
    messages.success(request, f'Dependent Removed Succesfully!')

    return redirect('view_profile')

# Password Change
def change_password(request):
    if request.method == 'POST':
        form = PasswordChangeForm(request.user, request.POST)
        if form.is_valid():
            user = form.save()
            update_session_auth_hash(request, request.user)
            messages.success(request, 'Your password was successfully changed!')
            return redirect('view_profile')
        else:
            messages.error(request, 'Please correct the error below.')
    else:
        form = PasswordChangeForm(request.user)
    return render(request, 'accounts/password_change.html', {
        'form': form
    })

# Deactivate Account
def deactivate_account(request):
    try:
        user = request.user
        user.is_active = False
        user.save()
        messages.success(request, 'Your account was deleted successfully!')
    except User.DoesNotExist: 
        messages.error(request, 'This user does not exist.')

    return redirect('logout')