import pytz
import datetime
from datetime import datetime, timedelta
from django.conf import settings
from django.utils import timezone
from django.http import JsonResponse
from django.forms.models import model_to_dict
from django.core.mail import send_mail
from django.views.generic import TemplateView
from django.shortcuts import render, redirect, get_object_or_404
from django.views.decorators.csrf import csrf_exempt
from django.contrib.auth.decorators import login_required
from django.contrib import messages
from django.contrib.auth.models import User
from motionsensor.models import MotionSensor
from motionsensor.forms import ( addMSForm,
                                    editMSForm )
from firedetector.models import FireDetector
from firedetector.forms import ( addFDForm,
                                    editFDForm )
from hvac.models import HVAC
from hvac.forms import ( addHVACForm,
                            editHVACForm )
from lights.models import Switch
from lights.forms import ( addSwitchForm,
                            editSwitchForm )
from outlets.models import Outlets
from outlets.forms import ( addOutletForm,
                            editOutletForm )
from accounts.models import Dependents
from friendship.models import Friend, Follow, Block, FriendshipRequest

def index(request):
    """Dashboard Home Page"""

    if not request.user.is_authenticated:
            messages.warning(request, 'Please Log In First.')
            return redirect('%s?next=%s' % (settings.LOGIN_URL, request.path))
    else:
        # Master Devices
        switches = Switch.objects.filter(owner=request.user.id)
        outlets = Outlets.objects.filter(owner=request.user.id)
        hvac = HVAC.objects.filter(owner=request.user.id)
        firedetector = FireDetector.objects.filter(owner=request.user.id)
        motionsensor = MotionSensor.objects.filter(owner=request.user.id)
        
        list = Dependents.objects.filter(person=request.user.id)
        if list:
            is_dependent = True
        else:
            is_dependent = False
            
        # Dependent Devices
        try:
            master = Friend.objects.friends(request.user)[0].id
            has_dependent = True
        except:
            has_dependent = False
        else:
            hvac_dependent = HVAC.objects.filter(owner=master)
            switches_dependent = Switch.objects.filter(owner=master)
            outlets_dependent = Outlets.objects.filter(owner=master)
            ms_dependent = MotionSensor.objects.filter(owner=master)
            fd_dependent = FireDetector.objects.filter(owner=master)        
        
        request_count = Friend.objects.unrejected_request_count(user=request.user)

        if request_count != 0:
            messages.info(request, 'You have %s dependent request(s) pending.' % request_count)

        # Email Notifications
        try:
            user_email = request.user.email
            if motionsensor:
                if motionsensor[0].status == "Activity Detected":
                    messages.error(request, 'Motion Sensor Triggered!')
                    send_mail('Motion Sensor Triggered', 'Attention, your Motion Sensor was triggered!', 'no-reply@hsas.site', [user_email,])
        except:
            print('Error sending email')
        try:
            user_email = request.user.email
            if firedetector:
                if firedetector[0].status == "Fire Hazard Detected":
                    messages.error(request, 'Fire Hazard Detected!')
                    send_mail('Fire Hazard Detected', 'Attention! Fire Hazard Detected!', 'no-reply@hsas.site', [user_email,])
        except:
            print('Error sending email')
        
        if has_dependent:
            context_dependent = {
                'switches': switches, 
                'outlets': outlets,
                'hvac': hvac,
                'firedetector': firedetector,
                'motionsensor': motionsensor,
                'hvac_dependent': hvac_dependent,
                'switches_dependent': switches_dependent,
                'outlets_dependent': outlets_dependent,
                'ms_dependent': ms_dependent,
                'fd_dependent': fd_dependent,
                'is_dependent': is_dependent,
            }
            return render(request, 'dashboard/home.html', context_dependent)
        else:
            context = {
                'switches': switches, 
                'outlets': outlets,
                'hvac': hvac,
                'firedetector': firedetector,
                'motionsensor': motionsensor,
            }
            return render(request, 'dashboard/home.html', context)


# HVAC Views

def create_hvac(request):
    """Add HVAC Form"""

    new_hvac_form = addHVACForm(request.POST or None)

    if new_hvac_form.is_valid():
        hvac = new_hvac_form.save(commit=False)
        hvac.owner = request.user
        new_hvac_form.save()
        return redirect('home')

    context = {'new_hvac_form': new_hvac_form}

    return render(request, 'dashboard/devices/hvac/add_hvac.html', context)

def update_hvac(request, sn):
    hvac = HVAC.objects.get(sn=sn)
    form = editHVACForm(request.POST or None, instance=hvac)

    if form.is_valid():
        form.save()
        return redirect('home')

    context = {
        'form': form,
        'hvac': hvac
    }

    return render(request, 'dashboard/devices/hvac/edit_hvac.html', context)

def toggle_hvac(request, sn):
    """Change HVAC Mode Form"""

    hvac = HVAC.objects.get(sn=sn)
    if hvac.cooling == 0:
        hvac.cooling = 1
        hvac.save()
    else:
        hvac.cooling = 0
        hvac.save()
    return redirect('home')

@login_required
@csrf_exempt
def delete_hvac(request, sn):
    """Delete HVAC"""

    HVAC.objects.get(sn=sn).delete()
    return redirect('home')

# Switch Views

def create_switch(request):
    """Add Switch Form"""

    new_switch_form = addSwitchForm(request.POST or None)

    if new_switch_form.is_valid():
        switch = new_switch_form.save(commit=False)
        switch.owner = request.user
        new_switch_form.save()
        return redirect('home')

    context = {'new_switch_form': new_switch_form}

    return render(request, 'dashboard/devices/switch/add_switch.html', context)

def update_switch(request, sn):
    switch = Switch.objects.get(sn=sn)
    form = editSwitchForm(request.POST or None, instance=switch)

    if form.is_valid():
        form.save()
        return redirect('home')

    context = {
        'form': form,
        'switch': switch
    }

    return render(request, 'dashboard/devices/switch/edit_switch.html', context)

@login_required
@csrf_exempt
def delete_switch(request, sn):
    """Delete Switch Form"""

    Switch.objects.get(sn=sn).delete()
    return redirect('home')

def toggle_switch(request, sn):
    """Add Switch Form"""

    switch = Switch.objects.get(sn=sn)
    if switch.status == 0:
        switch.status = 1
        messages.info(request, 'Switch Toggled')
        switch.save()
    elif switch.status == 1:
        switch.status = 0
        messages.info(request, 'Switch Toggled')
        switch.save()
    return redirect('home')

# Outlet Views

def create_outlet(request):
    """Add Outlet Form"""

    new_outlet_form = addOutletForm(request.POST or None)

    if new_outlet_form.is_valid():
        outlet = new_outlet_form.save(commit=False)
        outlet.owner = request.user
        new_outlet_form.save()
        return redirect('home')

    context = {'new_outlet_form': new_outlet_form}

    return render(request, 'dashboard/devices/outlet/add_outlet.html', context)

def update_outlet(request, sn):
    """Edit Outlet Form"""

    outlet = Outlets.objects.get(sn=sn)
    form = editOutletForm(request.POST or None, instance=outlet)

    if form.is_valid():
        form.save()
        return redirect('home')

    context = {
        'form': form,
        'outlet': outlet
    }

    return render(request, 'dashboard/devices/outlet/edit_outlet.html', context)

def toggle_outlet(request, sn):
    """Toggle Outlet"""

    outlet = Outlets.objects.get(sn=sn)
    if outlet.status == 0:
        outlet.status = 1
        messages.info(request, 'Outlet Turned On')
        outlet.save()
    else:
        outlet.status = 0
        messages.info(request, 'Outlet Turned Off')
        outlet.save()
    return redirect('home')

@login_required
@csrf_exempt
def delete_outlet(request, sn):
    """Delete Outlet Form"""

    Outlets.objects.get(sn=sn).delete()
    return redirect('home')

# Fire Detector Views

def create_fd(request):
    """Add Fire Detector Form"""

    new_fd_form = addFDForm(request.POST or None)

    if new_fd_form.is_valid():
        fd = new_fd_form.save(commit=False)
        fd.owner = request.user
        new_fd_form.save()
        return redirect('home')

    context = {'new_fd_form': new_fd_form}

    return render(request, 'dashboard/devices/firedetector/add_firedetector.html', context)

def update_fd(request, sn):
    """Update Fire Detector Form"""

    fd = FireDetector.objects.get(sn=sn)
    form = editFDForm(request.POST or None, instance=fd)

    if form.is_valid():
        form.save()
        return redirect('home')

    context = {
        'form': form,
        'fd': fd
    }

    return render(request, 'dashboard/devices/firedetector/edit_firedetector.html', context)

def toggle_fd(request, sn):
    """Fire Detector Toggle Form - Testing"""

    fd = FireDetector.objects.get(sn=sn)
    if fd.status == "OK":
        fd.status = "Fire Hazard Detected"
        fd.time_triggered = timezone.now()
        fd.save()
    else:
        fd.status = "OK"
        fd.save()
    return redirect('home')

@login_required
@csrf_exempt
def delete_fd(request, sn):
    """Delete Fire Detector Form"""

    FireDetector.objects.get(sn=sn).delete()
    return redirect('home')

# Motion Sensor Views

def create_ms(request):
    """Add Motion Sensor Form"""

    new_ms_form = addMSForm(request.POST or None)

    if new_ms_form.is_valid():
        ms = new_ms_form.save(commit=False)
        ms.owner = request.user
        new_ms_form.save()
        return redirect('home')

    context = {'new_ms_form': new_ms_form}

    return render(request, 'dashboard/devices/motionsensor/add_motionsensor.html', context)

def update_ms(request, sn):
    """Edit Motion Sensor Form"""

    ms = MotionSensor.objects.get(sn=sn)
    form = editMSForm(request.POST or None, instance=ms)

    if form.is_valid():
        form.save()
        return redirect('home')

    context = {
        'form': form,
        'ms': ms
    }

    return render(request, 'dashboard/devices/motionsensor/edit_motionsensor.html', context)

def toggle_ms(request, sn):
    """Edit Motion Sensor Form"""

    ms = MotionSensor.objects.get(sn=sn)
    if ms.status == "No Activity":
        ms.status = "Activity Detected"
        ms.time_triggered = timezone.now()
        ms.save()
    else:
        ms.status = "No Activity"
        ms.save()
    return redirect('home')

@login_required
@csrf_exempt
def delete_ms(request, sn):
    """Delete Motion Sensor Form"""

    MotionSensor.objects.get(sn=sn).delete()
    return redirect('home')