/*
  Copyright 2007, 2008 Computer Vision Lab,
  Ecole Polytechnique Federale de Lausanne (EPFL), Switzerland.
  All rights reserved.

  Authors: Vincent Lepetit (http://cvlab.epfl.ch/~lepetit)
           Mustafa Ozuysal (http://cvlab.epfl.ch/~oezuysal)
           Julien  Pilet   (http://cvlab.epfl.ch/~jpilet)

  This file is part of the ferns_demo software.

  ferns_demo is free software; you can redistribute it and/or modify it under the
  terms of the GNU General Public License as published by the Free Software
  Foundation; either version 2 of the License, or (at your option) any later
  version.

  ferns_demo is distributed in the hope that it will be useful, but WITHOUT ANY
  WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
  PARTICULAR PURPOSE. See the GNU General Public License for more details.

  You should have received a copy of the GNU General Public License along with
  ferns_demo; if not, write to the Free Software Foundation, Inc., 51 Franklin
  Street, Fifth Floor, Boston, MA 02110-1301, USA
*/
#include "tclap/CmdLine.h"
#include <cv.h>
#include <highgui.h>

#include <iostream>
#include <string>
using namespace std;
using namespace TCLAP;

#include "mcv.h"
#include "planar_pattern_detector_builder.h"
#include "template_matching_based_tracker.h"

#ifdef BOOST_THREAD_POOL
#define BOOST_FUNCTION_MAX_ARGS 20
#include "threadpool.hpp"
#include <boost/function.hpp>
#include <boost/bind.hpp>
using namespace boost;
using namespace boost::threadpool;
#endif

#if WIN32
#define snprintf sprintf_s
#endif

const int max_filename = 1000;

enum source_type {webcam_source, sequence_source, video_source};

planar_pattern_detector * detector = 0;

int mode = 2;
bool show_tracked_locations = true;
bool show_keypoints = true;

CvFont font;

void draw_matching_recognized_keypoints(IplImage * output, IplImage * frame, planar_pattern_detector * detector)
{
	float model_x = 0, model_y = 0, rec_x = 0, rec_y = 0;
	detector->detect(frame);	//	Detect stuff
	if (detector->pattern_is_detected) {
		for(int i = 0; i < detector->number_of_model_points; i++) {
			if (detector->model_points[i].class_score > 0) {
				model_x = detector->model_points[i].fr_u();
				model_y = detector->model_points[i].fr_v();
				rec_x = detector->model_points[i].potential_correspondent->fr_u() + detector->model_image->width;
				rec_y = detector->model_points[i].potential_correspondent->fr_v();
				cvCircle(output, cvPoint((int)model_x, (int)model_y), 1, CV_RGB(255, 0, 0), 2);
				cvCircle(output, cvPoint((int)rec_x, (int)rec_y), 1, CV_RGB(0, 0, 255), 2);
				cvLine(output, cvPoint((int)model_x, (int)model_y), cvPoint((int)rec_x, (int)rec_y), CV_RGB(255, 242-(i+2), 0), 1, CV_AA);
			}
		}
	}
}

int main(int argc, char ** argv)
{
	#ifdef TIMER
	clock_t start, end;
	double elapsed = 0;
	FILE *timer_output = fopen("planar_pattern_detector_builder_build_with_cache.txt", "w");
	#endif

	source_type frame_source = webcam_source;

	#ifdef BOOST_THREAD_POOL
	function<planar_pattern_detector * (builder_args *)> my_func (&planar_pattern_detector_builder::build_with_cache_pool);
	fifo_pool tp(4);
	#endif

	//	Begin parsing the command line arguments
	try {
		CmdLine cmd("This application is a demo to illustrate how to train a model.  Then use that model to detect the model image in a series of unrelated images, or video stream.  It detects points and displays those points and matches them to points it believes are from the model.", ' ', "1.0.0", true);

		ValueArg<string> model_image("m", string("model"), string("Specify the name of the model image depicting the planar object from a frontal viewpoint. Default model.bmp"), false, string("model.bmp"), string("string"), cmd);
		ValueArg<string> sequence("s", string("imgSeq"), string("Image sequence in printf style, e.g. image%04.jpg to test detection. If not specified webcam is used as image source."), false, string(""), string("string"), cmd);
		ValueArg<string> video("v", string("video"), string("Video filename to test detection. If not specified webcam is used as image source."), false, string(""), string("string"), cmd);

		cmd.parse(argc, argv);	//	Parse the command line arguments

		//	Set what source should be used for detection
		if(!sequence.getValue().empty())
			frame_source = sequence_source;
		else if(!video.getValue().empty())
			frame_source = video_source;

		affine_transformation_range range;
		#ifdef TIMER
		start = clock();
		fprintf(timer_output, "Running build with cache...\n");
		#endif

		detector = planar_pattern_detector_builder::build_with_cache(model_image.getValue().c_str(),
																	&range,
																	400,
																	5000,
																	0.0,
																	32, 7, 4,
																	50, 12,
																	10000, 200);
		#ifdef TIMER
		end = clock();
		elapsed = ((double) (end - start)) / CLOCKS_PER_SEC;
		fprintf(timer_output, "\t\tTime elapsed: %08.3fs\n", elapsed);
		fclose(timer_output);
		#endif

		/*builder_args test(model_image.c_str(),
							&range,
							400,
							5000,
							0.0,
							32, 7, 4,
							30, 12,
							10000, 200);

		tp.schedule(bind(my_func, &test));
		tp.schedule(bind(my_func, &test));
		tp.wait();*/

		if (!detector) {
			cerr << "Unable to build detector.\n";
			return -1;
		}

		detector->set_maximum_number_of_points_to_detect(1000);

		cvInitFont(&font, CV_FONT_HERSHEY_SIMPLEX, 1.0, 1.0, 0.0, 3, 8);

		CvCapture * capture = 0;
		IplImage * frame, * gray_frame = 0, * model_buf = 0, * debug = 0;
		int frame_id = 1;
		char seq_buffer[max_filename];
		char file_name[max_filename];

		model_buf = mcvGrayToColor(detector->model_image);

		debug = cvCreateImage(cvSize(model_buf->width*2, model_buf->height), IPL_DEPTH_8U, 3);

		cvNamedWindow("point-detector-demo", 1 );

		if(frame_source == webcam_source) {
			capture = cvCaptureFromCAM(0);
		} else if(frame_source == video_source) {
			capture = cvCreateFileCapture(video.getValue().c_str());
		}


		int64 timer = cvGetTickCount();

		bool stop = false;
		do {
			if(frame_source == webcam_source || frame_source == video_source) {
				if (cvGrabFrame(capture) == 0) break;
				frame = cvRetrieveFrame(capture);
			} else {
				snprintf(seq_buffer, max_filename, sequence.getValue().c_str(), frame_id);
				snprintf(file_name, max_filename, "./output/output_merge_%03d.bmp", frame_id);
				frame = cvLoadImage(seq_buffer, 1);
				++frame_id;
			}

			if (frame == 0) break;

			if (gray_frame == 0)
				gray_frame = cvCreateImage(cvSize(frame->width,frame->height), IPL_DEPTH_8U, 1);

			cvCvtColor(frame, gray_frame, CV_RGB2GRAY);

			if (frame->origin != IPL_ORIGIN_TL)
				cvFlip(gray_frame, gray_frame, 0);

			//	Put the model and the
			cvSetImageROI(debug, cvRect(0,0,model_buf->width, model_buf->height));
			cvCopy(model_buf, debug);
			cvSetImageROI(debug, cvRect(model_buf->width,0,model_buf->width, model_buf->height));
			cvCopy(frame, debug);
			cvResetImageROI(debug);

			//	Draw a vertical line between the two images
			cvLine(debug, cvPoint(model_buf->width, 0), cvPoint(model_buf->width, model_buf->height), CV_RGB(0, 255, 255), 4, 8);
			draw_matching_recognized_keypoints(debug, gray_frame, detector);
			//mcvSaveImage(file_name, debug);
			cvShowImage("point-detector-demo", debug);

			int64 now = cvGetTickCount();
			double fps = 1e6 * cvGetTickFrequency()/double(now-timer);
			timer = now;
			clog << "Detection frame rate: " << fps << " fps         \r";

			int key = cvWaitKey(10);
			if (key >= 0) {
				switch(char(key)) {
					case '0': mode = 0; break;
					case '1': mode = 1; break;
					case '2': mode = 2; break;
					case '3': mode = 3; break;
					case '4': show_tracked_locations = !show_tracked_locations; break;
					case '5': show_keypoints = !show_keypoints; break;
					case 'q': stop = true; break;
					default: ;
				}
				cout << "mode=" << mode << endl;
			}

			if(frame_source == sequence_source) {
				cvReleaseImage(&frame);
			}
		} while(!stop);

		clog << endl;
		delete detector;

		cvReleaseImage(&model_buf);	//	Release the model image
		cvReleaseImage(&debug);		//	Release the output image
		cvReleaseImage(&gray_frame);
		cvReleaseCapture(&capture);
		cvDestroyWindow("point-detector-demo");

		return 0;
	} catch (ArgException &e) {
		cerr << "error: " << e.error() << " for arg " << e.argId() << endl;
	}
}
