/*
  Copyright 2007 Computer Vision Lab,
  Ecole Polytechnique Federale de Lausanne (EPFL), Switzerland.
  All rights reserved.

  Author: Vincent Lepetit (http://cvlab.epfl.ch/~lepetit)

  This file is part of the ferns_demo software.

  ferns_demo is free software; you can redistribute it and/or modify it under the
  terms of the GNU General Public License as published by the Free Software
  Foundation; either version 2 of the License, or (at your option) any later
  version.

  ferns_demo is distributed in the hope that it will be useful, but WITHOUT ANY
  WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
  PARTICULAR PURPOSE. See the GNU General Public License for more details.

  You should have received a copy of the GNU General Public License along with
  ferns_demo; if not, write to the Free Software Foundation, Inc., 51 Franklin
  Street, Fifth Floor, Boston, MA 02110-1301, USA
*/
#ifndef planar_pattern_detector_builder_h
#define planar_pattern_detector_builder_h

#ifdef TIMER
#include <time.h>
#endif

#ifdef BOOST_THREAD_POOL
#include "threadpool.hpp"
#include <boost/function.hpp>
using namespace boost;
using namespace boost::threadpool;
#endif

#include <vector>
using namespace std;

#include "planar_pattern_detector.h"
#include "affine_transformation_range.h"

struct builder_args {
	const char * image_name;
	affine_transformation_range * range;
    int maximum_number_of_points_on_model;
    int number_of_generated_images_to_find_stable_points;
    double minimum_number_of_views_rate;
    int patch_size;
	int yape_radius;
	int number_of_octaves;
	int number_of_ferns;
	int number_of_tests_per_fern;
    int number_of_samples_for_refinement;
	int number_of_samples_for_test;
    char * given_detector_data_filename;
    int roi_up_left_u;
	int roi_up_left_v;
    int roi_bottom_right_u;
	int roi_bottom_right_v;
	builder_args(const char * Image_name,
			    affine_transformation_range * Range,
			    int Maximum_number_of_points_on_model,
			    int Number_of_generated_images_to_find_stable_points,
			    double Minimum_number_of_views_rate,
			    int Patch_size, int Yape_radius, int Number_of_octaves,
			    int Number_of_ferns, int Number_of_tests_per_fern,
			    int Number_of_samples_for_refinement, int Number_of_samples_for_test,
			    char * Given_detector_data_filename = 0,
			    int Roi_up_left_u = -1, int Roi_up_left_v = -1,
			    int Roi_bottom_right_u = -1, int Roi_bottom_right_v = -1)
	{
		image_name = Image_name;
		range = Range;
		maximum_number_of_points_on_model = Maximum_number_of_points_on_model;
		number_of_generated_images_to_find_stable_points = Number_of_generated_images_to_find_stable_points;
		minimum_number_of_views_rate = Minimum_number_of_views_rate;
		patch_size = Patch_size;
		yape_radius = Yape_radius;
		number_of_octaves = Number_of_octaves;
		number_of_ferns = Number_of_ferns;
		number_of_tests_per_fern = Number_of_tests_per_fern;
		number_of_samples_for_refinement = Number_of_samples_for_refinement;
		number_of_samples_for_test = Number_of_samples_for_test;
		given_detector_data_filename = Given_detector_data_filename;
		roi_up_left_u = Roi_up_left_u;
		roi_up_left_v = Roi_up_left_v;
		roi_bottom_right_u = Roi_bottom_right_u;
		roi_bottom_right_v = Roi_bottom_right_v;
	};
};

class planar_pattern_detector_builder
{
 public:
  //! Build the model, with caching.
  //! \return 0 on failure
	static planar_pattern_detector * build_with_cache_pool(builder_args * args);
  static planar_pattern_detector * build_with_cache(const char * image_name,
						    affine_transformation_range * range,
						    int maximum_number_of_points_on_model,
						    int number_of_generated_images_to_find_stable_points,
						    double minimum_number_of_views_rate,
						    int patch_size, int yape_radius, int number_of_octaves,
						    int number_of_ferns, int number_of_tests_per_fern,
						    int number_of_samples_for_refinement, int number_of_samples_for_test,
						    char * given_detector_data_filename = 0,
						    int roi_up_left_u = -1, int roi_up_left_v = -1,
						    int roi_bottom_right_u = -1, int roi_bottom_right_v = -1);

  static planar_pattern_detector * force_build(const char * image_name,
					       affine_transformation_range * range,
					       int maximum_number_of_points_on_model,
					       int number_of_generated_images_to_find_stable_points,
					       double minimum_number_of_views_rate,
					       int patch_size, int yape_radius, int number_of_octaves,
					       int number_of_ferns, int number_of_tests_per_fern,
					       int number_of_samples_for_refinement, int number_of_samples_for_test,
					       char * given_detector_data_filename = 0,
					       int roi_up_left_u = -1, int roi_up_left_v = -1,
					       int roi_bottom_right_u = -1, int roi_bottom_right_v = -1);

  static planar_pattern_detector * just_load(const char * given_detector_data_filename);

  //private:
  static planar_pattern_detector * learn(const char * image_name,
					 affine_transformation_range * range,
					 int maximum_number_of_points_on_model,
					 int number_of_generated_images_to_find_stable_points,
					 double minimum_number_of_views_rate,
					 int patch_size, int yape_radius, int number_of_octaves,
					 int number_of_ferns, int number_of_tests_per_fern,
					 int number_of_samples_for_refinement, int number_of_samples_for_test,
					 int roi_up_left_u = -1, int roi_up_left_v = -1,
					 int roi_bottom_right_u = -1, int roi_bottom_right_v = -1);


  static void detect_most_stable_model_points(planar_pattern_detector * detector,
					      int maximum_number_of_points_on_model,
					      int /*yape_radius*/, int /*number_of_octaves*/,
					      int number_of_generated_images,
					      double minimum_number_of_views_rate);

  static pair<keypoint, int> * search_for_existing_model_point(vector< pair<keypoint, int> > * tmp_model_points,
							       float cu, float cv, int scale);
  static pair<keypoint, int> * search_for_existing_model_point(vector< pair<keypoint, int> > * tmp_model_points, 
								   float cu, float cv, int scale, double* elapsed);

#ifdef BOOST_THREAD_POOL
  static void view_generators_thread_pool(boost::mutex* m_mutex, int i, int K, 
									int maximum_number_of_points_on_model, planar_pattern_detector * detector, 
									vector< pair<keypoint, int> > * tmp_model_point_vector);
#else
  static void view_generators_thread_pool(int i, int K, 
									int maximum_number_of_points_on_model, planar_pattern_detector * detector, 
									vector< pair<keypoint, int> > * tmp_model_point_vector);
#endif
};

#endif
