function MaxOVSIs = GreedySearch(OpenDSSFileName,DSSCircuit,DSSText,DSSSolution,...
    DSSElem,DSSBus)
    %% Intitialize variables
    MaxOVSIs = [];
    DesiredPercent = 89.781;                                                % Set desired percent of node voltages over/under voltage threshold
    
    % Set desired load scenario
    LoadScenario = 4;                                                       % 1 = Min, 2 = Midday Min, 3 = Midday Max, 4 = Max
    
    % Set PVScenario to "Large" if PV-to-three-phase connection is of
    % interest or "Small" if PV-to-service-transformer connections is of
    % interest
    PVScenario = "Large";                                           
    
    %% Set load multiplier based on desired load scenario
    % Change global load multiplier based on load scenario of interest
    % Loadmult only works on variable loads, not fixed loads
    if LoadScenario == 1
        DSSSolution.Loadmult = 0.24;
    elseif LoadScenario == 2
        DSSSolution.Loadmult = 0.51;
    elseif LoadScenario == 3
        DSSSolution.Loadmult = 0.79;
    elseif LoadScenario == 4
        DSSSolution.Loadmult = 1;
    else
        disp('Please enter a valid load scenario');
    end
    
    %% Set and lock regulator settings
    % Set regulator taps for load scenario of interest...these tap values
    % are found by running simulation with no PV and letting voltage
    % regulator operate to steady state
    % Please note that the "vreg" setting for regcontrol.creg4abc has been
    % changed from 124 to 120 as voltage profile seemed higher than
    % necessary
    LockVoltageRegs(LoadScenario,DSSCircuit,DSSText,DSSSolution,...
        DSSElem,DSSBus);
    
    % Lock taps
    RegControls = DSSCircuit.RegControls.First;
    while RegControls > 0
        DSSCircuit.RegControls.MaxTapChange = 0;
        RegControls = DSSCircuit.RegControls.Next;
    end
    
    
    %% Establish PVs at base levels
    alpha = 1;
    if logical(PVScenario == "Large")
        % Determine the number of three-phase buses...for large-scale PV
        % scenarios only
        BusNames = DSSCircuit.AllBusNames;
        ThreePhaseCount = 0;
        for i = 1:DSSCircuit.NumBuses
            DSSCircuit.SetActiveBus(char(BusNames(i)));
            if (DSSCircuit.ActiveBus.NumNodes == 3) && ...                  % Record each three-phase, primary bus
                    (DSSCircuit.ActiveBus.kVBase > 2.4)
                ThreePhaseCount = ThreePhaseCount + 1;
            end
        end

        % Establish PV such that 100% of load is distributed evenly among all
        % PVs
        Penetration = alpha/ThreePhaseCount;
        EstablishLargeScaleVariablePV(Penetration,DSSCircuit,DSSText,...
            DSSSolution,DSSElem,DSSBus);
    elseif logical(PVScenario == "Small")
        EstablishSmallScalePV(DSSCircuit,DSSText,DSSSolution,DSSElem,...
            DSSBus);
    else
        disp('Please use "Large" or "Small" to select PV scenario of interest');
    end
    AllPVNames = DSSCircuit.PVSystems.AllNames;
    AllPVNames = strcat("PVSystem.",AllPVNames);
    
    % Enable all PVs
    for i = 1:length(AllPVNames)
        DSSCircuit.SetActiveElement(AllPVNames(i));
        DSSCircuit.Enable(AllPVNames(i));
    end
    
    [SumVoltageDev SumVoltage OverUnderVoltagePercent] = ...
        SingleSolve(DSSCircuit,DSSText,DSSSolution,DSSElem,DSSBus);
    if DSSSolution.Converged == 0
        disp('Solution did not converge')
    end

    %% If over/under voltage percent is less than desired, adjust alpha until desired level is acheived    
    % If not enough nodes are over voltage threshold, keep increasing
    % penetration until there are. Once it is, perform greedy search
    % at that alpha level
    while OverUnderVoltagePercent < DesiredPercent
        % Start simulation fresh...without any defined PV
        DSSText.Command = strcat("Compile (",OpenDSSFileName,")");
        DSSText.Command = 'New Energymeter.reg1a transformer.reg1a terminal=1';
        DSSText.Command = 'Buscoords Buscoords.dat';
        
        % Change global load multiplier based on load scenario of interest
        % Loadmult only works on variable loads, not fixed loads
        if LoadScenario == 1
        DSSSolution.Loadmult = 0.24;
        elseif LoadScenario == 2
            DSSSolution.Loadmult = 0.51;
        elseif LoadScenario == 3
            DSSSolution.Loadmult = 0.79;
        elseif LoadScenario == 4
            DSSSolution.Loadmult = 1;
        else
            disp('Please enter a valid load scenario');
        end
        
        % Set regulator taps for load scenario of interest...these tap values
        % are found by running simulation with no PV and letting voltage
        % regulator operate to steady state
        % Please note that the "vreg" setting for regcontrol.creg4abc has been
        % changed from 124 to 120 as voltage profile seemed higher than
        % necessary
        LockVoltageRegs(LoadScenario,DSSCircuit,DSSText,DSSSolution,...
            DSSElem,DSSBus);
        
        % Lock taps
        RegControls = DSSCircuit.RegControls.First;
        while RegControls > 0
            DSSCircuit.RegControls.MaxTapChange = 0;
            RegControls = DSSCircuit.RegControls.Next;
        end

        % Re-establish PV at new penetration level
        alpha = alpha + 0.005;
        if logical(PVScenario == "Large")
            Penetration = alpha/ThreePhaseCount;
            EstablishLargeScaleVariablePV(Penetration,DSSCircuit,...
                DSSText,DSSSolution,DSSElem,DSSBus);
        elseif logical(PVScenario == "Small")
            EstablishSmallScaleVariablePV(alpha,DSSCircuit,DSSText,...
                DSSSolution,DSSElem,DSSBus);
        else
            disp('Please use "Large" or "Small" to select PV scenario of interest');
        end

        % Enable all PVs
        for i = 1:length(AllPVNames)
            DSSCircuit.SetActiveElement(AllPVNames(i));
            DSSCircuit.Enable(AllPVNames(i));
        end

        % Solve and see if percent of over/under voltages is desirable
        [SumVoltageDev SumVoltage OverUnderVoltagePercent] = ...
            SingleSolve(DSSCircuit,DSSText,DSSSolution,DSSElem,DSSBus);
        if DSSSolution.Converged == 0
            disp('Solution did not converge')
        end
        
        disp(['Current percent of nodes over voltage threshold is: ',num2str(OverUnderVoltagePercent)]);
    end

    %% Perform greedy-search algorithm and plot results
    while length(AllPVNames) >= 1;
        disp([num2str(length(AllPVNames)),' PVs are left']);
        OVSIs = [];

        % Remove one PV at a time, calculating the OVSI value each time
        for i = 1:length(AllPVNames)
            DSSCircuit.SetActiveElement(AllPVNames(i));
            DSSCircuit.Disable(AllPVNames(i));
            AllPVNames(i);

            [SumVoltageDev SumVoltage OverUnderVoltagePercent] = ...
                SingleSolve(DSSCircuit,DSSText,DSSSolution,DSSElem,DSSBus);
            if DSSSolution.Converged == 0
                disp('Solution did not converge')
            end
            OVSIPVName = string(AllPVNames(i));
            OVSIs = [OVSIs; OVSIPVName SumVoltageDev];

            DSSCircuit.SetActiveElement(AllPVNames(i));
            DSSCircuit.Enable(AllPVNames(i));
        end
        % Find the PV that results in the largest increase or smallest
        % decrease in OVSI when removed
        MaxOVSIValue = max(str2double(OVSIs(:,2)));
        MaxOVSIPVIndex = find(OVSIs(:,2)==string(MaxOVSIValue));
        MaxOVSIPV = OVSIs(MaxOVSIPVIndex(1),1);
        MaxOVSIs = [MaxOVSIs; MaxOVSIPV string(MaxOVSIValue)];

        % Permenantly remove that PV from future consideration
        DSSCircuit.SetActiveElement(MaxOVSIPV);                         % Remove PV of worst-case OVSI and perform removing sweep operation
        DSSCircuit.Disable(MaxOVSIPV);
        AllPVNamesIndex = find(AllPVNames == MaxOVSIPV);
        AllPVNames(AllPVNamesIndex) = [];
    end
    
    AllPVNames = DSSCircuit.PVSystems.AllNames;
    scatter(1:length(AllPVNames),str2double(MaxOVSIs(:,2)));
    xlabel('Number of PVs removed');
    ylabel('OVSI');
end